package com.xebialabs.xltest.bol;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.repository.TestRunsRepository;

public class DashboardCombinationCreator {

    private DashboardCombination.Builder builder;
    TestRunsRepository testRunsRepository;

    public DashboardCombinationCreator(TestRunsRepository testRunsRepository) {
        this.testRunsRepository = testRunsRepository;
    }

    public DashboardCombination getDashboardCombination(TestRun testRun, Map queryParameters) {
        return getDashboardCombination(testRun, queryParameters, false);
    }

    public DashboardCombination getDashboardCombination(TestRun testRun, Map queryParameters, boolean includeDates) {
        Date startTime = DateUtils.truncate(testRun.getStartTime(), Calendar.DATE);
        Date endTime = DateUtils.addDays(startTime, 1);

        List<Event> finished_events = getEvents(testRun, queryParameters, startTime, endTime);

        // get test run ids
        Set<String> runIds = new HashSet<>();
        for (Event finishEvent : finished_events) {
            runIds.add(finishEvent.getTestRunId().toString());
        }

        List<Event> result_events = new ArrayList<>();
        Map<String, Object> eventProperties = new HashMap();
        eventProperties.put("type", "result");
// TODO:       eventProperties.put("environment", testRun.getProperty("environment"));
        copyParameter(eventProperties, queryParameters, "team");
//        for (String theRunId : runIds) {
//            eventProperties.put("runId", theRunId);
//            List<Event> eventsInRunInTimeframe = testRun.getEventsBetweenInRunId(startTime.getTime(), endTime.getTime(), eventProperties, theRunId);

        List<Event> eventsInRunInTimeframe = testRunsRepository.getEventsBetween(startTime.getTime(), endTime.getTime(), eventProperties);
            for (Event event : eventsInRunInTimeframe) {
                if (matchingQueryParameters(event, queryParameters)) {
                    result_events.add(event);
                }
            }
//        }

        List<Event> timeoutsAndFailures = new ArrayList<>();
        for (Event finishEvent : finished_events) {
            if (!isSuccessEvent(finishEvent) && matchingQueryParameters(finishEvent, queryParameters)) {
                timeoutsAndFailures.add(finishEvent);
            }
        }

        builder = DashboardCombination.builder().setEvents(aggregateResults(result_events, timeoutsAndFailures));
        if (includeDates) {
            builder.setPreviousDate(determineDate(startTime, -1, testRun, queryParameters)).setNextDate(determineDate(startTime, +1, testRun, queryParameters));
        }
        return builder.build();
    }

    private List<Event> getEvents(TestRun testRun, Map queryParameters, Date startTime, Date endTime) {
        Map <String, Object> eventProperties = new HashMap();
        eventProperties.put("type", "jobStatus");
        eventProperties.put("status", "finished");

// TODO:       eventProperties.put("environment", testRun.getProperty("environment"));
        copyParameter(eventProperties, queryParameters, "team");

        return testRunsRepository.getEventsBetween(startTime.getTime(), endTime.getTime(), eventProperties);
    }

    private String determineDate(Date date, int i, TestRun testRun, Map queryParameters) {
        Date newDate = date;
        Date twoWeeksEarlier = DateUtils.addWeeks(date, -2);
        Date twoWeeksLater = DateUtils.addWeeks(date, 2);

        do {
            newDate = DateUtils.addDays(newDate, i);

        } while (newDate.before(twoWeeksLater) && newDate.after(twoWeeksEarlier) && getEvents(testRun, queryParameters, newDate, DateUtils.addDays(newDate, 1)).size() == 0);

        if (newDate.before(twoWeeksLater) && newDate.after(twoWeeksEarlier)) {
            return DateFormatUtils.format(newDate, "yyyy-MM-dd");
        }

        return null;
    }

    private void copyParameter(Map<String, Object> eventProperties, Map queryParameters, String key) {
        if (queryParameters.containsKey(key)) {
            eventProperties.put(key, queryParameters.get(key));
        }
    }

    private boolean matchingQueryParameters(Event event, Map<String, Object> queryParameters) {
        boolean match = true;

        if (event.hasProperty("team") && queryParameters.containsKey("team")) {
            match &= event.get("team").equals(queryParameters.get("team"));
        }

        if (event.hasProperty("testset") && queryParameters.containsKey("testset")) {
            match &= event.get("testset").equals("FrontPage." + queryParameters.get("testset")) ||
                    ((String) event.get("testset")).startsWith("FrontPage." + queryParameters.get("testset") + ".");
        }

        return match;
    }

    private boolean isSuccessEvent(Event finishEvent) {
        return "success".equals(finishEvent.get("reason"));
    }

    private List<Event> aggregateResults(List<Event> events, List<Event> timeoutsAndFailures) {
        Map<String, Event> agg = new HashMap<>();
        String keyAttr = "testset";

        for (Event e : events) {
            if (e.hasProperty(keyAttr)) {
                String key = e.get(keyAttr);
                Event other = agg.get(key);
                if (other == null || (Long) other.get("timestamp") < (Long) e.get("timestamp")) {
                    agg.put(key, e);
                }

                // if a later run of a usecase ran into timeout, mark the page as "timeout" by replacing the test result event by the
                // timeout event. Note this will happen for ALL test results in the usecase.
                for (Event to : timeoutsAndFailures) {
                    // Note we need to add a '.' otherwise
                    if (key.startsWith(to.get(keyAttr) + ".") && (Long) to.get("timestamp") > (Long) e.get("timestamp")) {
                        agg.put(key, to);
                        break;
                    }
                }
            }
        }

        for (Event event:timeoutsAndFailures) {
            if (event.hasProperty(keyAttr)) {
                final String key = event.get(keyAttr);
                Map<String, Event> filteredKeys = Maps.filterKeys(agg, new Predicate<String>() {
                    @Override
                    public boolean apply(String input) {
                        return input.startsWith(key + ".");
                    }
                });

                if (filteredKeys.size() == 0) {
                    agg.put(key, event);
                }
            }
        }

        return sortEvents(Lists.newArrayList(Sets.newHashSet(agg.values())), keyAttr);
    }

    private List<Event> sortEvents(List<Event> events, final String sortingProperty) {
        List<Event> filteredEvents = Lists.newArrayList(Iterables.filter(events, new Predicate<Event>() {
            @Override
            public boolean apply(Event e) {
                return e.hasProperty(sortingProperty) && e.get(sortingProperty) instanceof Comparable;
            }
        }));

        Collections.sort(filteredEvents, new Comparator<Event>() {
            @Override
            public int compare(Event o1, Event o2) {
                String s1 = o1.get(sortingProperty);
                String s2 = o2.get(sortingProperty);
                return s1.compareTo(s2);
            }});
        return filteredEvents;
    }

}
