package com.xebialabs.xlplatform.watcher

import com.xebialabs.xlplatform.watcher.Existence.Existence
import com.xebialabs.xlplatform.watcher.StateChange.StateChange
import grizzled.slf4j.Logging

import java.io.File

abstract class FileWatch(fileToWatch: File) extends Logging {

  private var lastState: State = currentState

  type Notify = StateChange => Unit

  private def currentState = State(currentFileExistence(), fileToWatch.lastModified)

  private def currentFileExistence() = if (fileToWatch.exists) {
    Existence.Present
  } else {
    Existence.NotPresent
  }

  def doWhenModified(doNotify: Notify): Unit = {
    trace(s"Polling changes on file ${fileToWatch.getPath}")
    val newState = currentState
    if (isStateChanged(newState)) {
      val change = stateChange(newState)
      debug(s"State of the file ${fileToWatch.getPath} changed to $change")
      if (change == StateChange.NotChanged) {
        warn(s"State of the file ${fileToWatch.getPath} not expected to be $change")
      } else {
        doNotify(change)
      }
      lastState = newState
    }
  }

  private def isStateChanged(newState: State) = newState != lastState

  private def stateChange(newState: State) = lastState.existence match {
    case Existence.NotPresent =>
      newState.existence match {
        case Existence.Present => StateChange.Created
        case Existence.NotPresent => StateChange.NotChanged
      }
    case Existence.Present =>
      newState.existence match {
        case Existence.Present =>
          if (lastState.modified != newState.modified) StateChange.Modified else StateChange.NotChanged
        case Existence.NotPresent =>
          StateChange.Deleted
      }
  }

  private case class State(existence: Existence, modified: Long)

}

private[watcher] object Existence extends Enumeration {
  type Existence = Value
  val Present, NotPresent = Value
}

object StateChange extends Enumeration {
  type StateChange = Value
  val Created, Modified, Deleted, NotChanged = Value
}
