package com.xebialabs.xlplatform.utils

import java.io.{Closeable, InputStream, OutputStream}

trait ManagedResource[T] {
  def onEnter(): T
  def onExit(exceptionHappened: Boolean = false): Unit
  def attempt(block: => Unit): Unit = {
    try { block } finally {}
  }
}


class ManagedClosable[C <: Closeable](closable: C) extends ManagedResource[C] {
  def onEnter(): C = closable
  def onExit(exceptionHappened: Boolean): Unit = {
    attempt(closable.close())
  }
}


class ManagedOutputStream(out:OutputStream) extends ManagedResource[OutputStream] {
  def onEnter(): OutputStream = out
  def onExit(exceptionHappened: Boolean): Unit = {
    attempt(out.close())
  }
}

class ManagedInputStream(in:InputStream) extends ManagedResource[InputStream] {
  def onEnter(): InputStream = in
  def onExit(exceptionHappened: Boolean): Unit = {
    attempt(in.close())
  }
}

object ResourceManagement {
  def map[T <: Any, A <: Any](managed: ManagedResource[T])(body: T => A): A = {
    val resource = managed.onEnter()
    var exception = false
    try {
      body(resource)
    } catch {
      case e: Throwable =>
        exception = true
        managed.onExit(exceptionHappened = true)
        throw e
    } finally {
      if (!exception) managed.onExit()
    }
  }

  def using[T <: Any, A](managed: ManagedResource[T])(body: T => A) = {
    map(managed)(body)
  }

  def using[T <: Any, U <: Any](managed1: ManagedResource[T], managed2: ManagedResource[U])(body: T => U => Unit) {
    using[T, Unit](managed1) { r: T =>
      using[U, Unit](managed2) { s: U => body(r)(s) }
    }
  }

  implicit def os2managed[T<:OutputStream](out:T): ManagedResource[OutputStream] = new ManagedOutputStream(out)
  implicit def is2managed(in:InputStream): ManagedResource[InputStream] = new ManagedInputStream(in)
  implicit def managedClosable[T <: Closeable](closable:T): ManagedResource[T] = new ManagedClosable[T](closable)
}
