package com.xebialabs.xlplatform.ui

import java.net.URL

import scala.xml.{NodeSeq, Node, Elem}
import com.xebialabs.xlplatform.utils.ResourceUtils

sealed trait IMenu extends Ordered[IMenu] {
  def weight: Int
  def label: String

  override def compare(that: IMenu): Int = {
    if (this.weight == that.weight) {
      this.label.compareTo(that.label)
    } else {
      this.weight - that.weight
    }
  }
}
case class MenuSeparator(weight: Int) extends IMenu {
  def label = "--------"
}

case class MenuItem(weight: Int, label: String, uri: String) extends IMenu
case class Menu(id: String, weight: Int, label: String, uri: Option[String], items: List[IMenu]) extends IMenu

object UiMenus {
  import ResourceUtils._

  def apply(filePattern: String): List[Menu] = {

    val uiPluginElements: Map[URL, Elem] = loadXmlResources(filePattern)

    def applyReferences(menus: List[IMenu], elems: NodeSeq): List[Menu] = {
      val lookup: Map[String, Menu] = menus.collect({
        case m: Menu => (m.id, m)
      }).toMap

      var notFoundMenuIds = List[String]()

      val resultedMenus = (elems \ "menu-ref").foldLeft(lookup)({
        case (l, n) =>
          val menuId: String = (n \ "@ref").text
          if (l.contains(menuId)) {
            val appendedMenu: Menu = appendItems(l(menuId), n)
            val appendedWithRefs = appendedMenu.copy(items = (appendedMenu.items.filterNot(_.isInstanceOf[Menu]) ++ applyReferences(appendedMenu.items, n)).sorted)
            l + ((menuId, appendedWithRefs))
          } else {
            notFoundMenuIds = menuId :: notFoundMenuIds
            l
          }
      }).values.toList

      if (notFoundMenuIds.nonEmpty) {
        throw new NotFoundException(s"Menus with the next IDs are not found: ${notFoundMenuIds mkString ", "}")
      }

      resultedMenus
    }

    def getMenuItem(e: Node): MenuItem = MenuItem((e \ "@weight").text.toInt, (e \ "@label").text, (e \ "@uri").text)
    def getMenu(e: Node): Menu = appendItems(Menu((e \ "@id").text, (e \ "@weight").text.toInt, (e \ "@label").text, (e \ "@uri").headOption.map(_.text), List()), e)

    def getMenuSeparator(e: Node): MenuSeparator = MenuSeparator((e \ "@weight").text.toInt)

    def appendItems(m: Menu, e: Node): Menu = m.copy(items = (m.items ++ e.child.collect({
      case node: Elem if node.label == "menu-item" => getMenuItem(node)
      case node: Elem if node.label == "menu" => getMenu(node)
      case node: Elem if node.label == "menu-separator" => getMenuSeparator(node)
    })).sorted)


    def checkMenuIdsOnDuplication(uiPluginElements: Map[URL, Elem]): Unit = {
      val allLevelMenuIds: Map[URL, Seq[String]] = uiPluginElements.mapValues(e => (e \\ "menu").map(p => (p \ "@id").text))

      def findDuplicates[A](xs: Seq[A]): Seq[A] = xs.distinct.filter(x => xs.count(_ == x) > 1)

      def findURLsById(id: String): Seq[URL] = allLevelMenuIds.toSeq.filter(_._2.contains(id)).map(_._1)

      val duplicates: Seq[(String, Seq[URL])] = findDuplicates(allLevelMenuIds.values.flatten.toSeq).map {
        id: String => (id, findURLsById(id))
      }

      if (duplicates.nonEmpty)
        throw new DuplicationException(duplicates)
    }


    checkMenuIdsOnDuplication(uiPluginElements)
    val menus: List[IMenu] = uiPluginElements.values.toList.flatMap(e => e \ "menu").map(getMenu)

    applyReferences(menus, uiPluginElements.values.flatten.toSeq)
  }


}
