package com.xebialabs.deployit.security;

import java.util.Collection;
import java.util.HashSet;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;

import com.xebialabs.deployit.security.authentication.AuthenticationFailureException;
import com.xebialabs.deployit.security.authentication.PersonalAuthenticationToken;
import com.xebialabs.deployit.security.permission.Permission;

public class Permissions {
    public static Authentication getAuthentication() {
        return SecurityContextHolder.getContext().getAuthentication();
    }

    public static String getAuthenticatedUserName() {
        return getAuthentication() != null ? getAuthentication().getName() : null;
    }

    public static Collection<String> authenticationToPrincipals(Authentication authentication) {
        if (authentication == null) {
            throw new AuthenticationFailureException();
        }
        final String name = authentication.getName();
        final Collection<String> allPrincipals = authentication.getAuthorities().stream().map(GrantedAuthority::getAuthority).collect(Collectors.toCollection(HashSet::new));
        allPrincipals.add(name);
        logger.trace("All principals for user [{}] are: {}", authentication.getName(), allPrincipals);
        return allPrincipals;
    }

    public static void clearSecurityContext() {
        SecurityContextHolder.clearContext();
    }

    private static final Logger logger = LoggerFactory.getLogger(Permissions.class);

    public static Collection<Permission> isApplicableTo(Collection<Permission> values, final String id) {
        return isApplicableTo(values, id, true);
    }

    public static Collection<Permission> isApplicableTo(Collection<Permission> values, final String id, boolean checkRoot) {
        return values.stream().filter(p -> !p.isApplicableTo(id, checkRoot)).collect(Collectors.toList());
    }

    public static boolean isPersonalAccessToken() {
        return getAuthentication() instanceof PersonalAuthenticationToken;
    }

    public static boolean isLegacyPersonalAccessToken() {
        return isPersonalAccessToken() && ((PersonalAuthenticationToken) getAuthentication()).isLegacyToken();
    }
}
