package com.xebialabs.deployit.security.authentication;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Set;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.Transient;
import org.springframework.util.Assert;

@Transient
public class PersonalAuthenticationToken extends AbstractAuthenticationToken {

    private final Object principal;

    private String token;

    private Date expiryDate;

    /**
     * Fine-grained global permissions scope for personal authentication token. Empty means, all permissions are assigned.
     */
    private final Set<String> globalPermissions;

    /**
     * Fine-grained permissions on CI level for personal authentication token. Empty means, all permissions are assigned.
     */
    private final Set<String> configurationItemsPermissions;

    /**
     * Set of CIs for which configurationItemsPermissions are applied
     */
    private final Set<String> configurationItems;

    public PersonalAuthenticationToken(String token) {
        super(Collections.emptyList());
        Assert.hasText(token, "token cannot be empty");
        this.principal = null;
        this.token = token;
        this.globalPermissions = Collections.unmodifiableSet(Collections.emptySet());
        this.configurationItemsPermissions = Collections.unmodifiableSet(Collections.emptySet());
        this.configurationItems = Collections.unmodifiableSet(Collections.emptySet());
        setAuthenticated(false);
    }

    public PersonalAuthenticationToken(Object principal,
                                       String token,
                                       Date expiryDate,
                                       Collection<? extends GrantedAuthority> authorities,
                                       Set<String> globalPermissions,
                                       Set<String> configurationItemsPermissions,
                                       Set<String> configurationItems) {
        super(authorities);
        this.principal = principal;
        this.token = token;
        this.expiryDate = expiryDate;
        this.globalPermissions = Collections.unmodifiableSet(globalPermissions);
        this.configurationItemsPermissions = Collections.unmodifiableSet(configurationItemsPermissions);
        this.configurationItems = Collections.unmodifiableSet(configurationItems);
        setAuthenticated(true);
    }

    @Override
    public Object getCredentials() {
        return this.token;
    }

    @Override
    public Object getPrincipal() {
        return this.principal;
    }

    @Override
    public void eraseCredentials() {
        super.eraseCredentials();
        this.token = null;
    }

    public Date getExpiryDate() {
        return expiryDate;
    }

    public Set<String> getGlobalPermissions() {
        return globalPermissions;
    }

    public Set<String> getConfigurationItemsPermissions() {
        return configurationItemsPermissions;
    }

    public Set<String> getConfigurationItems() {
        return configurationItems;
    }

    public boolean isLegacyToken() {
        return globalPermissions.isEmpty() && configurationItemsPermissions.isEmpty();
    }
}

