package com.xebialabs.deployit.security;

import java.util.*;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.authentication.PersonalAuthenticationToken;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PlatformPermissions;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.deployit.security.Permissions.authenticationToPrincipals;
import static com.xebialabs.deployit.security.Permissions.getAuthentication;
import static java.util.stream.Collectors.toList;

@Component
public class PermissionEnforcer {

    public static final String ROLE_ADMIN = "ROLE_ADMIN";

    private PermissionChecker checker;
    private final RoleService roleService;

    @Autowired
    public PermissionEnforcer(PermissionChecker checker, RoleService roleService) {
        this.checker = checker;
        this.roleService = roleService;
    }

    public boolean hasLoggedInUserPermission(final Permission... permissions) {
        Authentication authentication = getAuthentication();
        return hasPermission(authentication, Arrays.asList(permissions), "");
    }

    public boolean hasLoggedInUserPermission(Permission permission, String onConfigurationItem) {
        return hasLoggedInUserPermission(Collections.singletonList(permission), onConfigurationItem);
    }

    public boolean hasLoggedInUserPermission(List<Permission> permissions, String onConfigurationItem) {
        return hasPermission(getAuthentication(), permissions, onConfigurationItem);
    }

    public Map<String, Boolean> hasLoggedInUserPermission(Permission permission, List<String> onConfigurationItems) {
        return hasLoggedInUserPermission(Collections.singletonList(permission), onConfigurationItems);
    }

    public Map<String, Boolean> hasLoggedInUserPermission(List<Permission> permissions, List<String> onConfigurationItems) {
        return hasPermission(getAuthentication(), permissions, onConfigurationItems);
    }

    public void applyLoggedInUserPermission(SearchParameters parameters, Permission permission) {
        applyLoggedInUserPermission(parameters, Collections.singletonList(permission));
    }

    public void applyLoggedInUserPermission(SearchParameters parameters, List<Permission> permissions) {
        applyPermission(parameters, getAuthentication(), permissions);
    }

    public boolean hasPermission(Authentication authentication, final Permission... permissions) {
        return hasPermission(authentication, Arrays.asList(permissions), "");
    }

    public boolean hasPermission(Authentication auth, List<Permission> permissions, String onConfigurationItem) {
        boolean hasRequiredScope = hasRequiredTokenPermissionScope(auth, permissions, onConfigurationItem);

        if (hasRequiredScope) {
            final Collection<String> allPrincipals = authenticationToPrincipals(auth);
            List<Role> allRoles = getUserRoles(auth);
            //scope check is already done for user, don't trigger it again with admin permission check
            return isAdmin(allPrincipals, allRoles, Permissions.getAuthentication(), false) || hasPermission(allRoles, permissions, onConfigurationItem);
        } else {
            return false;
        }
    }

    public Map<String, Boolean> hasPermission(Authentication auth, List<Permission> permissions, List<String> onConfigurationItems) {
        List<String> itemsWithRequiredScope = new ArrayList<>();
        Map<String, Boolean> allPermissions = new HashMap<>();

        onConfigurationItems.forEach(item -> {
            if (hasRequiredTokenPermissionScope(auth, permissions, item)) {
                itemsWithRequiredScope.add(item);
            } else {
                allPermissions.put(item, false);
            }
        });

        final Collection<String> allPrincipals = authenticationToPrincipals(auth);
        List<Role> allRoles = getUserRoles(auth);
        //scope check is already done for user, don't trigger it again with admin permission check
        if (isAdmin(allPrincipals, allRoles, Permissions.getAuthentication(), false)) {
            itemsWithRequiredScope.forEach(item -> allPermissions.put(item, true));
        } else {
            Map<String, Boolean> requiredScopePermissions = hasPermission(allRoles, permissions, itemsWithRequiredScope);
            allPermissions.putAll(requiredScopePermissions);
        }
        return allPermissions;
    }

    public void applyPermission(SearchParameters parameters, Authentication auth, List<Permission> permissions) {
        final Collection<String> allPrincipals = authenticationToPrincipals(auth);
        List<Role> allRoles = getUserRoles(auth);
        if (!isAdmin(allPrincipals, allRoles)) {
            applyPermission(parameters, allRoles, permissions);
        }
    }

    public List<Role> getUserRoles(Authentication auth) {
        return roleService.getRolesFor(auth);
    }

    public boolean hasPermission(List<Role> roles, List<Permission> permissions, String onConfigurationItem) {
        return checker.checkPermission(permissions, onConfigurationItem, roles);
    }

    public Map<String, Boolean> hasPermission(List<Role> roles, List<Permission> permissions, List<String> onConfigurationItems) {
        return checker.checkPermission(permissions, onConfigurationItems, roles);
    }

    public void applyPermission(SearchParameters parameters, List<Role> roles, List<Permission> permissions) {
        parameters.setSecurityParameters(
                roles.stream().map(Role::getId).collect(toList()),
                permissions.stream().map(Permission::getPermissionName).collect(toList())
        );
    }

    private static final Logger logger = LoggerFactory.getLogger(PermissionEnforcer.class);

    public boolean isCurrentUserAdmin() {
        return isAdmin(getAuthentication());
    }

    public boolean isAdmin(Authentication auth) {
        Collection<String> principals = authenticationToPrincipals(auth);
        List<Role> allRoles = roleService.getRolesFor(auth);
        return isAdmin(principals, allRoles, auth, true);
    }

    public boolean isAdmin(Collection<String> allPrincipals, List<Role> allRoles) {
        return isAdmin(allPrincipals, allRoles, Permissions.getAuthentication(), true);
    }

    private boolean isAdmin(Collection<String> allPrincipals, List<Role> allRoles, Authentication auth, boolean withScopeCheck) {
        List<Permission> permissions = Collections.singletonList(PlatformPermissions.ADMIN);
        boolean hasRequiredScope = withScopeCheck ? hasRequiredTokenPermissionScope(auth, permissions, "") : true;
        if (hasRequiredScope) {
            boolean isAdmin = allPrincipals.contains(ROLE_ADMIN) || checker.checkPermission(permissions, "", allRoles, auth);
            logger.trace("Admin privileges [{}] granted to {}", isAdmin ? "are" : "are not", allPrincipals);
            return isAdmin;
        } else {
            return false;
        }
    }

    private boolean hasRequiredTokenPermissionScope(Authentication auth, List<Permission> permissions, String onConfigurationItem) {
        boolean hasRequiredScope = true;
        if (auth instanceof PersonalAuthenticationToken) {
            PersonalAuthenticationToken token = (PersonalAuthenticationToken) auth;
            Set<String> requiredPermissions = permissions.stream().map(Permission::getPermissionName).collect(Collectors.toSet());
            // check if it is fine-grained token or legacy
            boolean isLegacyToken = token.isLegacyToken();

            if (isLegacyToken) {
                hasRequiredScope = true;
            } else if (isNotEmpty(onConfigurationItem)) {
                if (token.getConfigurationItems().contains(onConfigurationItem)) {
                    hasRequiredScope = token.getConfigurationItemsPermissions().containsAll(requiredPermissions);
                } else {
                    hasRequiredScope = false;
                }
            } else {
                hasRequiredScope = token.getGlobalPermissions().containsAll(requiredPermissions);
            }
        }
        return hasRequiredScope;
    }
}
