package com.xebialabs.deployit.security.permission;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot;
import com.xebialabs.deployit.security.permission.PermissionHandler.Level;

import static com.xebialabs.deployit.security.permission.PermissionHandler.Level.BOTH;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Level.CI;
import static com.xebialabs.deployit.security.permission.PermissionHandler.Level.GLOBAL;

public class Permission {

    private static final Map<String, Permission> REGISTERED_PERMISSIONS = new LinkedHashMap<>();
    private static final List<Permission> IMPLICIT_READ_PERMISSIONS = new ArrayList<>();
    // Copy-pasted jcr specific field
    private static final String GLOBAL_SECURITY_ALIAS = "global";

    private final String permissionName;
    private final Level level;
    private final PermissionHandler permissionHandler;
    private final Metadata.ConfigurationItemRoot root;

    //
    // Constructor & factory methods
    //

    private Permission(String permissionName, Level level, PermissionHandler permissionHandler, ConfigurationItemRoot root) {
        this.permissionName = permissionName;
        this.level = level;
        this.root = root;

        if (permissionHandler != null) {
            this.permissionHandler = permissionHandler;
        } else {
            this.permissionHandler = new PermissionHandler(this);
        }
    }

    public static Permission definePermission(String name, Level level) {
        Permission permission = new Permission(name, level, null, null);
        registerPermission(permission, false);
        return permission;
    }

    public static Permission definePermission(String name, Level level, PermissionHandler handler, boolean implicitRead) {
        Permission permission = new Permission(name, level, handler, null);
        registerPermission(permission, implicitRead);
        return permission;
    }

    public static Permission definePermission(String name, Level level, Metadata.ConfigurationItemRoot root) {
        return definePermission(name, level, root, false);
    }

    public static Permission definePermission(String name, Level level, Metadata.ConfigurationItemRoot root, boolean implicitRead) {
        Permission permission = new Permission(name, level, null, root);
        registerPermission(permission, implicitRead);
        return permission;
    }

    //
    // Permission registry
    //

    private static void registerPermission(Permission permission, boolean implicitRead) {
        REGISTERED_PERMISSIONS.put(permission.getPermissionName(), permission);
        if (implicitRead) {
            IMPLICIT_READ_PERMISSIONS.add(permission);
        }
    }

    public static Permission find(String permissionName) {
        return REGISTERED_PERMISSIONS.get(permissionName);
    }

    public static Collection<Permission> getAll() {
        return REGISTERED_PERMISSIONS.values();
    }

    public static List<Permission> getReadPermissions() {
        return Collections.unmodifiableList(IMPLICIT_READ_PERMISSIONS);
    }

    //
    // Etcetera
    //

    public PermissionHandler getPermissionHandler() {
        return permissionHandler;
    }

    public String getPermissionName() {
        return permissionName;
    }

    public Level getLevel() {
        return level;
    }

    public Metadata.ConfigurationItemRoot getRoot() {
        return root;
    }

    public boolean isApplicableTo(String id) {
        return isApplicableTo(id, true);
    }

    public boolean isApplicableTo(String id, boolean checkRoot) {
        id = id != null ? id : "";
        boolean idIsGlobal = id.isEmpty() || id.equals(GLOBAL_SECURITY_ALIAS);
        if (level == GLOBAL && idIsGlobal) return true;
        if (level == BOTH && idIsGlobal) return true;
        if (level == CI && idIsGlobal) return false;

        if (level != BOTH && level != CI) {
            return false;
        }

        if (root == null) {
            return true;
        }

        return !checkRoot || id.startsWith(root.getRootNodeName());
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof Permission) {
            return permissionName.equals(((Permission) other).permissionName);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return permissionName.hashCode();
    }

    @Override
    public String toString() {
        if (level != GLOBAL && root != null) {
            return permissionName + " on " + root;
        }

        return permissionName;
    }
}
