package com.xebialabs.deployit.security.authentication;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.authority.mapping.GrantedAuthoritiesMapper;
import org.springframework.security.core.authority.mapping.NullAuthoritiesMapper;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.PermissionEnforcer;
import com.xebialabs.deployit.security.User;
import com.xebialabs.deployit.security.UserService;

@Component
public class XlAuthenticationProvider implements AuthenticationProvider {

    private static final Logger logger = LoggerFactory.getLogger(XlAuthenticationProvider.class);

    private UserService userService;

    private GrantedAuthoritiesMapper authoritiesMapper = new NullAuthoritiesMapper();

    @Autowired
    public XlAuthenticationProvider(UserService userService) {
        this.userService = userService;
    }

    @Override
    public Authentication authenticate(final Authentication token) throws AuthenticationException {
        logger.debug("Authenticating for XL product: {}", token.getPrincipal());
        final String username = token.getPrincipal().toString();
        final String password = token.getCredentials().toString();

        return doAuthentication(username, password);
    }

    private Authentication doAuthentication(final String username, final String password) {
        if (Strings.isBlank(username)) {
            throw new BadCredentialsException("Cannot authenticate with empty username");
        }

        try {
            logger.trace("Authenticating [{}]", username);
            userService.authenticate(username, password);
            User user = userService.read(username);

            List<GrantedAuthority> authorities = new ArrayList<>();
            if (user.isAdmin() || "admin".equals(user.getUsername())) {
                authorities.add(new SimpleGrantedAuthority(PermissionEnforcer.ROLE_ADMIN));
            }

            Collection<? extends GrantedAuthority> mappedAuthorities = authoritiesMapper.mapAuthorities(authorities);

            return new XlAuthenticationToken(username, password, mappedAuthorities);
        } catch (AuthenticationFailureException | NotFoundException exc) {
            throw new BadCredentialsException("Cannot authenticate " + username, exc);
        }
    }

    @Override
    public boolean supports(final Class<?> authentication) {
        return authentication.isAssignableFrom(XlAuthenticationToken.class);
    }

    public GrantedAuthoritiesMapper getAuthoritiesMapper() {
        return authoritiesMapper;
    }

    public void setAuthoritiesMapper(GrantedAuthoritiesMapper authoritiesMapper) {
        this.authoritiesMapper = authoritiesMapper;
    }

}
