package com.xebialabs.deployit.security.authentication;

import java.nio.charset.Charset;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import com.xebialabs.deployit.conversion.Base64Coder;
import com.xebialabs.deployit.engine.spi.event.LoginEvent;
import com.xebialabs.deployit.engine.spi.event.LogoutEvent;
import com.xebialabs.deployit.event.EventBusHolder;

import static com.xebialabs.deployit.security.Permissions.getAuthentication;


public class RememberMe {

    private static final Cache<String, Authentication> AUTH_CACHE = CacheBuilder.newBuilder().expireAfterAccess(10, TimeUnit.MINUTES).build();

    public static void remember(Authentication auth) {
        String key = createKey(auth);
        logger.debug("Remembering authentication token for [{}]", auth.getName());
        if (AUTH_CACHE.getIfPresent(key) == null) {
            EventBusHolder.publish(new LoginEvent(auth.getName()));
        }
        AUTH_CACHE.put(key, auth);
    }

    public static void forget(Authentication auth) {
        String key = createKey(auth);
        if (AUTH_CACHE.getIfPresent(key) != null) {
            AUTH_CACHE.invalidate(key);
            EventBusHolder.publish(new LogoutEvent(auth.getName()));
        } else {
            logger.info("Couldn't forget authentication for [{}] because it was not remembered.", auth.getName());
        }
    }

    public static Authentication getRemembered(String username, String password) {
        String key = encodeUserPassword(username, password);
        Authentication ifPresent = AUTH_CACHE.getIfPresent(key);
        if (ifPresent == null) {
            logger.debug("Token for [{}] not yet remembered.", username);
        }
        return ifPresent;
    }

    public static void forgetMe() {
        forget(getAuthentication());
    }

    private static String createKey(Authentication auth) {
        if (auth instanceof XlAuthenticationToken) {
            XlAuthenticationToken token = (XlAuthenticationToken) auth;
            String user = token.getName();
            String password = token.getCredentials().toString();
            return encodeUserPassword(user, password);
        } else {
            throw new IllegalArgumentException("The authentication should always be a XlAuthenticationToken");
        }
    }

    private static String encodeUserPassword(String user, String password) {
        return new String(new Base64Coder().encode((user + ":" + password).getBytes()), Charset.defaultCharset());
    }

    private static final Logger logger = LoggerFactory.getLogger(RememberMe.class);
}
