package com.xebialabs.deployit.security;

import java.util.Collection;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Strings;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.JcrPathHelper;
import com.xebialabs.deployit.repository.core.Securable;
import com.xebialabs.deployit.security.permission.Permission;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.security.Permissions.buildLookup;
import static com.xebialabs.deployit.security.Permissions.joinRoles;
import static com.xebialabs.deployit.security.Permissions.readPermissionMap;
import static com.xebialabs.deployit.security.Permissions.rolesToIds;
import static com.xebialabs.deployit.security.Permissions.splitRoles;
import static com.xebialabs.deployit.security.Permissions.writePermissionMap;

@Component
public class PermissionEditor {

    private final static Type SECURABLE = Type.valueOf(Securable.class);

    private final JcrTemplate jcrTemplate;
    private final RoleService roleService;

    @Autowired
    public PermissionEditor(JcrTemplate jcrTemplate, RoleService roleService) {
        this.jcrTemplate = jcrTemplate;
        this.roleService = roleService;
    }

    public Multimap<Role, Permission> readPermissions(String id) {
        final String absPath = getPath(id);
        logger.debug("Reading permissions from [{}]", absPath);

        return jcrTemplate.execute(new JcrCallback<Multimap<Role, Permission>>() {
            public Multimap<Role, Permission> doInJcr(Session session) throws RepositoryException {
                return internalReadPermissions(session, absPath);
            }
        });
    }

    public void editPermissions(String id, final Multimap<Role, Permission> permissions) {
        final String absPath = getPath(id);
        logger.debug("Going to write permissions {} to node [{}]", permissions, absPath);

        Collection<Permission> notApplicableTo = Permissions.isApplicableTo(permissions.values(), id);
        checkArgument(notApplicableTo.isEmpty(), "The permissions %s are not applicable to [%s]", notApplicableTo, id);

        jcrTemplate.execute(new JcrCallback<Object>() {
            public Object doInJcr(Session session) throws RepositoryException {
                internalWritePermissions(session, absPath, permissions);
                session.save();
                return null;
            }
        });
    }

    private Multimap<Role, Permission> internalReadPermissions(Session session, String absPath) throws RepositoryException {
        Checks.checkArgument(session.nodeExists(absPath), "Couldn't find a node at [%s]", JcrPathHelper.getIdFromAbsolutePath(absPath));
        Node node = session.getNode(absPath);
        checkType(node);
        final Multimap<Role, Permission> permissions = HashMultimap.create();
        final ImmutableMap<Integer,Role> lookup = buildLookup(roleService.getRoles());
        Map<String,String> permissionEntries = readPermissionMap(node);
        logger.debug("Found permission entries on node: {}", permissionEntries);
        for (Map.Entry<String, String> permissionRoles : permissionEntries.entrySet()) {
            Permission p = Permission.find(permissionRoles.getKey());
            for (Integer roleId : splitRoles(permissionRoles.getValue())) {
                if (lookup.containsKey(roleId)) {
                    permissions.put(lookup.get(roleId), p);
                }
            }
        }
        logger.debug("Read from [{}] permissions {}", absPath, permissions);
        return permissions;
    }

    private void checkType(Node node) throws RepositoryException {
        if (node.hasProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME)) {
            Type type = Type.valueOf(node.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString());
            checkArgument(type.instanceOf(SECURABLE), "The type [{}] does not support security permissions, because it doesn't implement com.xebialabs.deployit.repository.core.Securable", type);
        }
    }

    private void internalWritePermissions(Session session, String absPath, Multimap<Role, Permission> permissions) throws RepositoryException {
        Node node = session.getNode(absPath);
        Multimap<Permission, Role> permission2Roles = HashMultimap.create();
        Multimaps.invertFrom(permissions, permission2Roles);
        Map<String, String> map = newHashMap();
        for (Permission permission : permission2Roles.keySet()) {
            Collection<Role> roles = permission2Roles.get(permission);
            String value = joinRoles(rolesToIds(roles));
            logger.debug("Going to write permission-roles: [{} -> {}]", permission, value);
            map.put(permission.getPermissionName(), value);
        }
        writePermissionMap(node, map);
    }

    private String getPath(String id) {
        final String nodeId;
        if (Strings.nullToEmpty(id).isEmpty() || JcrConstants.GLOBAL_SECURITY_ALIAS.equals(id)) {
            nodeId = JcrConstants.SECURITY_NODE_ID;
        } else {
            nodeId = JcrPathHelper.getAbsolutePathFromId(id);
        }
        return nodeId;
    }

    private static final Logger logger = LoggerFactory.getLogger(PermissionEditor.class);
}
