package com.xebialabs.deployit.security;

import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Collections2;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.repository.core.Securable;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.filter;
import static com.xebialabs.deployit.jcr.JcrConstants.SECURITY_NODE_ID;
import static com.xebialabs.deployit.security.Permissions.readPermissionMap;
import static com.xebialabs.deployit.security.Permissions.readSecurable;
import static com.xebialabs.deployit.security.Permissions.rolesToIds;
import static com.xebialabs.deployit.security.Permissions.splitRoles;

@Component
public class PermissionLister {

    private final JcrTemplate jcrTemplate;
    private final RepositoryService repositoryService;

    @Autowired
    public PermissionLister(JcrTemplate jcrTemplate, RepositoryService repositoryService) {
        this.jcrTemplate = jcrTemplate;
        this.repositoryService = repositoryService;
    }

    public Multimap<String, String> listPermissions(Role role) {
        return listPermissions(newArrayList(role));
    }

    public Multimap<String, String> listPermissions(final List<Role> roles) {
        final HashMultimap<String, String> permissions = HashMultimap.create();
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                readGlobalPermissions(session, permissions, roles);
                readCiLevelPermissions(session, permissions, roles);
                return null;
            }
        });
        return permissions;
    }

    private void readCiLevelPermissions(Session session, final HashMultimap<String, String> permissionList, final List<Role> roles) throws RepositoryException {
        final List<ConfigurationItemData> cisWithPermissions = repositoryService.list(new SearchParameters().setType(Type.valueOf(Securable.class)));
        for (ConfigurationItemData ci : cisWithPermissions) {
            Node s = readSecurable(ci.getId(), session);
            readPermissionsFromNode(permissionList, roles, ci.getId(), s);
        }
    }

    private static void readPermissionsFromNode(HashMultimap<String, String> permissionList, final List<Role> roles, String ci, Node s) throws RepositoryException {
        Map<String, String> permissions = readPermissionMap(s);
        permissionList.putAll(ci, Collections2.transform(filter(permissions.entrySet(), new Predicate<Map.Entry<String, String>>() {
            public boolean apply(Map.Entry<String, String> input) {
                return checkAllowed(input.getValue(), roles);
            }
        }), new Function<Map.Entry<String, String>, String>() {
            public String apply(Map.Entry<String, String> input) {
                return input.getKey();
            }
        }));
    }

    private static void readGlobalPermissions(Session session, final HashMultimap<String, String> permissionList, final List<Role> roles) throws RepositoryException {
        Node securityNode = session.getNode(SECURITY_NODE_ID);
        readPermissionsFromNode(permissionList, roles, JcrConstants.GLOBAL_SECURITY_ALIAS, securityNode);
    }

    private static boolean checkAllowed(String allowedRoles, List<Role> principals) {
        Iterable<Integer> split = splitRoles(allowedRoles);
        return Iterables.any(split, Predicates.in(newArrayList(rolesToIds(principals))));
    }

}
