package com.xebialabs.deployit.security;

import com.google.common.base.Function;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.jcr.JcrUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.ROLES_NODE_ID;
import static com.xebialabs.deployit.jcr.JcrConstants.ROLE_ASSIGNMENTS_NODE_ID;
import static com.xebialabs.deployit.jcr.JcrUtils.*;
import static com.xebialabs.deployit.security.Permissions.*;

/**
 * Extracts and stores roles in JCR
 *
 * The Multimap&lt;String, String&gt; which is used to communicate is a mapping with:
 * - key: role name
 * - value: principals associated to the role
 * 
 * However the storage in JCR is:
 * - propertyname: principal
 * - propertyvalue: comma-separated roles
 * 
 * The reason for this is that it makes for a very fast lookup when logging in. 
 */
@Component
public class RoleServiceImpl implements RoleService {

    private static final AtomicInteger groupNumber = new AtomicInteger(-1);

    private JcrTemplate jcrTemplate;

    @Autowired
    public RoleServiceImpl(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
    public List<Role> getRoles() {
        return jcrTemplate.execute(new JcrCallback<List<Role>>() {
            public List<Role> doInJcr(Session session) throws RepositoryException {
                return readRoles(session);
            }
        });
    }

    @Override
    public List<Role> readRoleAssignments() {
        return jcrTemplate.execute(new JcrCallback<List<Role>>() {
            @Override
            public List<Role> doInJcr(Session session) throws RepositoryException {
                return readRoleAssignments(session);
            }
        });
    }

    @Override
    public void writeRoleAssignments(final List<Role> roles) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                writeRoles(session, roles);
                writeRoleAssignments(session, roles);
                session.save();
                return null;
            }
        });
    }

    @Override
    public Role getRoleForRoleName(final String roleName) {
        List<Role> roles = getRoles();
        for (Role role : roles) {
            if (role.getName().equals(roleName)) {
                return role;
            }
        }
        
        throw new NotFoundException("Could not find the role [%s]", roleName);
    }

    @Override
    public List<Role> getRolesFor(final Authentication auth) {
        final Collection<String> principals = Permissions.authenticationToPrincipals(auth);
        return getRolesFor(principals);
    }

    @Override
    public List<Role> getRolesFor(final String principal) {
        return getRolesFor(newArrayList(principal));
    }

    private List<Role> readRoles(Session session) throws RepositoryException {
        Node node = session.getNode(ROLES_NODE_ID);
        final List<Role> roles = newArrayList();
        forEachNonJcrProperty(node, new JcrUtils.Callback<Property>() {
            public void apply(Property input) throws RepositoryException {
                roles.add(new Role(Integer.valueOf(input.getName()), input.getString()));
            }
        });
        return roles;
    }

    private List<Role> readRoleAssignments(Session session) throws RepositoryException {
        Node node = session.getNode(ROLE_ASSIGNMENTS_NODE_ID);
        final List<Role> roles = getRoles();
        final Map<Integer, Role> lookup = buildLookup(roles);

        forEachNonJcrProperty(node, new JcrUtils.Callback<Property>() {
            public void apply(Property input) throws RepositoryException {
                String principal = input.getName();
                Iterable<Integer> roles = splitRoles(input.getString());
                for (Integer role : roles) {
                    lookup.get(role).getPrincipalsAssigned().add(principal);
                }
            }
        });
        return roles;
    }

    private void writeRoles(Session session, List<Role> roles) throws RepositoryException {
        Node node = session.getNode(ROLES_NODE_ID);

        if (groupNumber.get() == -1) {
            groupNumber.compareAndSet(-1, readMaxRole(node) + 1);
        }

        clearProperties(node);

        for (Role role : roles) {
            if ((role.getId() == null) || (role.getId() == -1)) {
                role.setId(groupNumber.getAndIncrement());
            }
            node.setProperty(role.getId().toString(), role.getName());
        }
    }

    private int readMaxRole(Node node) throws RepositoryException {
        final int[] i = {-1};
        forEachNonJcrProperty(node, new JcrUtils.Callback<Property>() {
            public void apply(Property input) throws RepositoryException {
                int j = Integer.valueOf(input.getName());
                if (j > i[0]) i[0] = j;
            }
        });
        return i[0];
    }

    private List<Role> getRolesFor(final Collection<String> principals) {
        return jcrTemplate.execute(new JcrCallback<List<Role>>() {
            public List<Role> doInJcr(Session session) throws RepositoryException {
                return principalsToRoles(session, principals);
            }
        });
    }

    private List<Role> principalsToRoles(Session session, Collection<String> principals) throws RepositoryException {
        Node node = session.getNode(ROLE_ASSIGNMENTS_NODE_ID);
        final ImmutableMap<Integer, Role> lookup = buildLookup(getRoles());
        Function<Integer, Role> roleIdToRole = new Function<Integer, Role>() {
            public Role apply(Integer input) {
                return lookup.get(input);
            }
        };

        List<Role> roles = newArrayList();
        for (String principal : principals) {
            principal = escapeIllegalJcrChars(principal);
            if (node.hasProperty(principal)) {
                Property property = node.getProperty(principal);
                roles.addAll(newArrayList(Iterables.transform(splitRoles(property.getString()), roleIdToRole)));
            }
        }
        logger.trace("Found roles {} for principals {}", roles, principals);
        return roles;
    }

    private void writeRoleAssignments(Session session, List<Role> map) throws RepositoryException {
        Node node = session.getNode(ROLE_ASSIGNMENTS_NODE_ID);
        clearProperties(node);

        ListMultimap<String, Role> principalRole = buildMap(map);
        
        for (String principal : principalRole.keySet()) {
            node.setProperty(escapeIllegalJcrChars(principal), joinRoles(Iterables.transform(principalRole.get(principal), new Function<Role, Integer>() {
                public Integer apply(Role input) {
                    return input.getId();
                }
            })));
        }
    }

    private ListMultimap<String, Role> buildMap(List<Role> map) {
        ArrayListMultimap<String, Role> principalToRoles = ArrayListMultimap.create();
        for (Role role : map) {
            for (String principal : role.getPrincipalsAssigned()) {
                principalToRoles.put(principal, role);
            }
        }
        return principalToRoles;
    }

    private static final Logger logger = LoggerFactory.getLogger(RoleServiceImpl.class);
}
