package com.xebialabs.deployit.security;

import java.util.List;

import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import org.hamcrest.CoreMatchers;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.security.core.Authentication;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.authentication.UserAlreadyExistsException;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

public abstract class UserServiceTest {
    
    private static String ADMIN_USERNAME = "admin";

    private static String ADMIN_PASSWORD = "admin";

    private static String TEST_USERNAME = "test-user";

    private static String TEST_PASSWORD = "initial-password-for-test-user";

    private static String MODIFIED_TEST_PASSWORD = "modified-password-for-test-user";

    private UserService userService;

    private Authentication originalAuthentication;

    protected abstract UserService getUserService();

    @Before
    public void setupUserService() {
        userService = getUserService();
    }

    @Before
    public void authenticateAsAdmin() {
        originalAuthentication = SecurityTemplate.setCredentials(ADMIN_USERNAME, ADMIN_PASSWORD);
    }

    @After
    public void restoreOriginalAuthentication() {
        SecurityTemplate.restoreAuthentication(originalAuthentication);
    }

    @Test
    public void canReadAdminUser() {
        User adminUser = userService.read(ADMIN_USERNAME);

        assertThat(adminUser, is(notNullValue()));
        assertThat(adminUser.getUsername(), CoreMatchers.is(ADMIN_USERNAME));
        assertThat(adminUser.isAdmin(), is(true));
    }

    @Test
    public void shouldListUsernames() {
        userService.create(TEST_USERNAME + "-1", TEST_PASSWORD);
        userService.create(TEST_USERNAME + "-2", TEST_PASSWORD);

        final List<String> usernames = userService.listUsernames();

        assertThat(usernames, hasItems(TEST_USERNAME + "-1", TEST_USERNAME + "-2", "admin"));
        assertThat(String.format("expecting 3 usernames, got these: [%s]", String.join(", ", usernames)), usernames, hasSize(3));

        userService.delete(TEST_USERNAME + "-1");
        userService.delete(TEST_USERNAME + "-2");
    }

    @Test
    public void shouldListFilteredUsernames() {
        userService.create(TEST_USERNAME + "-1", TEST_PASSWORD);
        userService.create(TEST_USERNAME + "-2", TEST_PASSWORD);
        userService.create("Vanya", TEST_PASSWORD);
        userService.create("Paul", TEST_PASSWORD);

        final List<String> usernames = userService.listUsernames("user", null, null);
        assertThat(usernames, hasItems("test-user-1", "test-user-2"));

        final List<String> usernames2 = userService.listUsernames("user", null, new Ordering("DESC"));
        assertThat(usernames2, hasItems("test-user-2", "test-user-1"));

        final Paging paging1 = new Paging(1, 3);
        final List<String> usernames3 = userService.listUsernames(null, paging1, new Ordering("DESC"));
        assertThat(usernames3, hasItems("Vanya", "test-user-2", "test-user-1"));
        assertThat(usernames3, hasSize(3));

        final Paging paging2 = new Paging(4, 1);
        final List<String> usernames4 = userService.listUsernames(null, paging2, new Ordering("DESC"));
        assertThat(usernames4, hasItems("Paul"));
        assertThat(usernames4, hasSize(1));

        final List<String> usernames5 = userService.listUsernames("test", paging1, new Ordering("userName:ASC"));
        assertThat(usernames5, hasSize(2));
        assertThat(usernames5, hasItems("test-user-1", "test-user-2"));

        userService.delete(TEST_USERNAME + "-1");
        userService.delete(TEST_USERNAME + "-2");
        userService.delete("Vanya");
        userService.delete("Paul");
    }

    @Test
    public void shouldCreateModifyAndDeleteUser() {
        shouldCreateUser();
        shouldModifyPasswordOfUser();
        shouldDeleteUser();
    }

    @Test
    public void shouldModifyPasswordsProvidingTheOldOne() {
        shouldCreateUser();
        shouldRejectModifyingPasswordWhenPreviousIsWrong();
        shouldModifyPasswordWhenProvidingPrevious();
        shouldDeleteUser();
    }

    @Test(expected = UserAlreadyExistsException.class)
    public void shouldRejectCreatingSameUserTwice() {
            userService.create("Paul", TEST_PASSWORD);

        try {
            userService.create("Paul", TEST_PASSWORD);
        } catch (UserAlreadyExistsException e) {
            userService.delete("Paul");
            throw e;
        }
    }

    @Test(expected = UserAlreadyExistsException.class)
    public void shouldRejectCreatingSameUserWithDifferentCaseTwice() {
        userService.create("Vanya", TEST_PASSWORD);

        try {
            userService.create("Vanya".toUpperCase(), TEST_PASSWORD);
        } catch (UserAlreadyExistsException e) {
            userService.delete("Vanya");
            throw e;
        }
    }

    private void shouldCreateUser() {
        assertTestUserDoesNotExist();

        userService.create(TEST_USERNAME, TEST_PASSWORD);

        assertTestUserDoesExist();
        assertTestUserCanLogin();
    }

    private void shouldModifyPasswordOfUser() {
        userService.modifyPassword(TEST_USERNAME, MODIFIED_TEST_PASSWORD);
        assertTestUserCanLogin();
    }

    private void shouldModifyPasswordWhenProvidingPrevious() {
        userService.modifyPassword(TEST_USERNAME, MODIFIED_TEST_PASSWORD, TEST_PASSWORD);
        assertTestUserCanLogin();
    }

    private void shouldRejectModifyingPasswordWhenPreviousIsWrong() {
        Exception caughtException = null;

        try {
            userService.modifyPassword(TEST_USERNAME, MODIFIED_TEST_PASSWORD, MODIFIED_TEST_PASSWORD);
        } catch (Exception exception) {
            caughtException = exception;
        }

        assertThat(caughtException, is(notNullValue()));
        assertThat(caughtException, is(instanceOf(IllegalArgumentException.class)));
        assertThat(caughtException.getMessage(), containsString("Failed to change password: Old password does not match."));
    }

    private void shouldDeleteUser() {
        assertThat(userService.read(TEST_USERNAME), is(notNullValue()));

        userService.delete(TEST_USERNAME);

        assertTestUserDoesNotExist();
    }

    private void assertTestUserDoesNotExist() {
        try {
            userService.read(TEST_USERNAME);
            fail("Did not expect to read user " + TEST_USERNAME);
        } catch (NotFoundException expectedException) {
            /* success! */
        }
    }

    private void assertTestUserDoesExist() {
        User foundUser = userService.read(TEST_USERNAME);
        assertThat(foundUser, is(notNullValue()));
        assertThat(foundUser.getUsername(), is(TEST_USERNAME));
        assertThat(foundUser.isAdmin(), is(false));
    }

    protected abstract void assertTestUserCanLogin();

}
