package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.PlatformPermissions._
import com.xebialabs.deployit.security.permission.{Permission, PermissionHandler, PlatformTestPermissions}
import com.xebialabs.xlplatform.test.UnitTestBase
import org.springframework.security.authentication.TestingAuthenticationToken

import scala.collection.JavaConverters._

trait PermissionEnforcerTest extends UnitTestBase {

  implicit val roleService: RoleService

  implicit val permissionChecker: PermissionChecker

  implicit val permissionEditor: PermissionEditor

  var loginHandler: PermissionHandler = _
  var readHandler: PermissionHandler = _

  var envGroup: Directory = _
  var envDept: Directory = _
  var environment: Environment = _
  var appGroup: Directory = _

  var devver: Role = _
  var admin: Role = _
  var lRoleA: Role = _
  var lRoleB: Role = _
  var lRoleC: Role = _

  lazy val permissions = new TestPermissions(roleService, permissionEditor)
  lazy val permissionEnforcer = new PermissionEnforcer(permissionChecker, roleService)

  def setup(): Unit = {
    new SecurityServiceLocator(permissionEnforcer, null)
    loginHandler = LOGIN.getPermissionHandler
    readHandler = READ.getPermissionHandler

    devver = new Role("devver")
    admin = new Role("administrator")
    lRoleA = new Role("A-Role").withRoles(devver.getName)
    lRoleB = new Role("B-Role").withPrincipals("user-1")
    lRoleC = new Role("C-Role").withRoles(devver.getName)
    devver.getPrincipals.add("group-1")
    admin.getPrincipals.add("admin-group")
    roleService.writeRoleAssignments(Seq(devver, admin).asJava)
    roleService.writeRoleAssignments(envGroup.getId, Seq(lRoleA).asJava)
    roleService.writeRoleAssignments(envDept.getId, Seq(lRoleB, lRoleC).asJava)
    // This took me some figuring out. The repository filters the securitypermissions unless you have the permission:
    SecurityTemplate.setAuthentication(new TestingAuthenticationToken("admin", "admin", "ROLE_ADMIN"))
    permissions.grant(EDIT_SECURITY, devver, null)
    SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "group-1"))
  }

  def cleanup(): Unit = {
    roleService.writeRoleAssignments(Seq.empty[Role].asJava)
  }

  describe("Permission enforcer") {
    it("should check for permission granted to role") {
      assert(!loginHandler.hasPermission(null.asInstanceOf[String]))

      permissions.grant(LOGIN, devver, null)
      assert(loginHandler.hasPermission(null))

      permissions.revoke(LOGIN, devver, null)
      assert(!loginHandler.hasPermission(null.asInstanceOf[String]))
    }

    it("should check for ci group permission granted to role on same node") {
      checkPermissions(devver, envGroup.getId, envGroup.getId)
    }

    it("should check for ci permission granted to role on same node") {
      checkPermissions(lRoleA, envGroup.getId, envGroup.getId)
    }

    it("should have read permission when granted ci level permission") {
      checkRelatedReadPermission(READ, envGroup.getId, envGroup.getId)
      checkRelatedReadPermission(PlatformTestPermissions.IMPLICIT_READ_APP, appGroup.getId, appGroup.getId)
    }

    it("should not revoked explicit read when revoking another permission") {
      permissions.grant(READ, devver, envGroup.getId)
      val deployHandler = PlatformTestPermissions.IMPLICIT_READ.getPermissionHandler
      permissions.grant(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId)
      assert(readHandler.hasPermission(envGroup.getId))
      assert(deployHandler.hasPermission(envGroup.getId))

      permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId)
      assert(!deployHandler.hasPermission(envGroup.getId))
      assert(readHandler.hasPermission(envGroup.getId))
      permissions.revoke(READ, devver, envGroup.getId)
    }

    it("should not revoke explicit read when revoking another permission from other role") {
      val editRepoHandler = EDIT_REPO.getPermissionHandler
      assert(!readHandler.hasPermission(envDept.getId))
      assert(!editRepoHandler.hasPermission(envDept.getId))

      permissions.grant(READ, lRoleC, envDept.getId)
      assert(readHandler.hasPermission(envDept.getId))
      assert(!editRepoHandler.hasPermission(envDept.getId))

      permissions.grant(EDIT_REPO, lRoleB, envDept.getId)
      assert(readHandler.hasPermission(envDept.getId))
      assert(editRepoHandler.hasPermission(envDept.getId))

      permissions.revoke(EDIT_REPO, lRoleB, envDept.getId)
      assert(readHandler.hasPermission(envDept.getId))
      assert(!editRepoHandler.hasPermission(envDept.getId))

      permissions.revoke(READ, lRoleC, envDept.getId)
      assert(!readHandler.hasPermission(envDept.getId))
      assert(!editRepoHandler.hasPermission(envDept.getId))
    }

    it("should not remove read permission for user when another user has granted related permission") {
      permissions.grant(READ, devver, envGroup.getId)
      // Yes, user-2 is here on purpose...
      permissions.grant(PlatformTestPermissions.IMPLICIT_READ, admin, envGroup.getId)
      permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId)
      assert(readHandler.hasPermission(envGroup.getId))

      permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, admin, envGroup.getId)
      permissions.revoke(READ, devver, envGroup.getId)
    }

    it("should be admin user when granted admin permission") {
      assert(!readHandler.hasPermission(envGroup.getId))
      permissions.grant(ADMIN, devver, null)
      assert(permissionEnforcer.isAdmin(Seq("user-1").asJava, Seq(devver).asJava))
      assert(readHandler.hasPermission(envGroup.getId))

      permissions.revoke(ADMIN, devver, null)
      assert(!readHandler.hasPermission(envGroup.getId))
    }

    it("should be admin user when granted correct authority") {
      SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "ROLE_ADMIN"))
      assert(permissionEnforcer.isAdmin(Seq("user-1", "ROLE_ADMIN").asJava, Seq.empty[Role].asJava))
      assert(readHandler.hasPermission(envGroup.getId))
    }

    it("should revoke permission in one go when granted twice") {
      permissions.grant(READ, devver, envGroup.getId)
      permissions.grant(READ, devver, envGroup.getId)
      assert(readHandler.hasPermission(envGroup.getId))
      permissions.revoke(READ, devver, envGroup.getId)
      assert(!readHandler.hasPermission(envGroup.getId))
    }

    it("should revoke permission when removing user from local role membership") {
      val lRoleD = new Role("D-Role")
      roleService.writeRoleAssignments(envDept.getId, Seq(lRoleB, lRoleD).asJava)
      assert(!readHandler.hasPermission(envDept.getId))

      permissions.grant(READ, lRoleA, envDept.getId)
      permissions.grant(READ, lRoleD, envDept.getId)
      assert(!readHandler.hasPermission(envDept.getId))

      permissions.grant(READ, lRoleB, envDept.getId)
      assert(readHandler.hasPermission(envDept.getId))

      lRoleB.withPrincipals("unknown")
      roleService.writeRoleAssignments(envDept.getId, Seq(lRoleB, lRoleD).asJava)
      assert(!readHandler.hasPermission(envDept.getId))

      lRoleB.withPrincipals("user-1")
      roleService.writeRoleAssignments(envDept.getId, Seq(lRoleB, lRoleD).asJava)
      assert(readHandler.hasPermission(envDept.getId))
    }
  }

  protected def checkPermissions(role: Role, onConfiguration: String, ci: String): Unit = {
    assert(!readHandler.hasPermission(onConfiguration))
    permissions.grant(READ, role, ci)
    assert(readHandler.hasPermission(onConfiguration))
    permissions.revoke(READ, role, ci)
    assert(!readHandler.hasPermission(onConfiguration))
  }

  protected def checkRelatedReadPermission(p: Permission, grantedNode: String, checkNode: String): Unit = {
    permissions.grant(p, devver, grantedNode)
    assert(readHandler.hasPermission(checkNode))
    assert(p.getPermissionHandler.hasPermission(checkNode))
    permissions.revoke(p, devver, grantedNode)
    assert(!p.getPermissionHandler.hasPermission(checkNode))
    assert(!readHandler.hasPermission(checkNode))
  }
}
