package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.security.{Role, RoleService}
import com.xebialabs.xlplatform.test.UnitTestBase

import scala.jdk.CollectionConverters._

trait CrudRoleService_GlobalTest extends UnitTestBase {
  implicit val roleService: RoleService

  val developerRole: Role = new Role("developer").withPrincipals("john")
  val deployerRole: Role = new Role("deployer").withPrincipals("john", "sam")
  val administratorRole: Role = new Role("administrator").withPrincipals("kate")

  var readRoleAssignmentsPaginationImplemented = false
  var countImplemented = false

  protected def setup(): Unit = {
    roleService.createOrUpdateRole(developerRole)
    roleService.createOrUpdateRole(deployerRole)
    roleService.createOrUpdateRole(administratorRole)
  }

  protected def tearDown(): Unit = {
    deleteAllRoles()
  }

  protected def deleteAllRoles(): Unit = {
    roleService.getRoles().forEach(r => roleService.deleteById(r.getId))
  }

  implicit class RoleWithNameAndPrincipal(role: Role) {
    def hasNameAndPrincipalsAs(checkRole: Role): Boolean =
      role.getName.equals(checkRole.getName) && role.getPrincipals.containsAll(checkRole.getPrincipals)

    def hasNameAs(checkRole: Role): Boolean =
      role.getName.equals(checkRole.getName)
  }

  describe("Role Service with global roles") {
    it("should not fail on no roles read") {
      deleteAllRoles()

      val rolePrincipalsMap = roleService.readRoleAssignments().asScala

      assert(rolePrincipalsMap != null)
      assert(rolePrincipalsMap.isEmpty)
    }

    it("should read written roles") {
      deleteAllRoles()

      val roles = roleService.getRoles().asScala

      assert(roles.isEmpty)

      roleService.create("developer")

      assert(roleService.getRoles().size() == 1)
    }

    it("should create a role and write role assignments") {
      val result = roleService.readRoleAssignments().asScala

      assert(result.length == 3)

      assert(result.exists(_.hasNameAndPrincipalsAs(developerRole)))
      assert(result.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(result.exists(_.hasNameAndPrincipalsAs(administratorRole)))
    }

    it("should update existing role principals with new") {
      deleteAllRoles()
      roleService.createOrUpdateRole(developerRole)

      val result = roleService.readRoleAssignments().asScala
      assert(result.length == 1)
      assert(result.head.hasNameAndPrincipalsAs(developerRole))

      val newDeveloperRole = new Role(developerRole.getName)
      roleService.createOrUpdateRole(newDeveloperRole)

      val resultAfterChange = roleService.readRoleAssignments().asScala
      assert(resultAfterChange.length == 1)
      assert(resultAfterChange.exists(r => r.getName.equals(developerRole.getName) && r.getPrincipals.isEmpty))
    }

    it("should remove role by name") {
      val roles = roleService.readRoleAssignments().asScala

      assert(roles.exists(_.hasNameAndPrincipalsAs(developerRole)))
      assert(roles.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(roles.exists(_.hasNameAndPrincipalsAs(administratorRole)))

      roleService.deleteByName(developerRole.getName)

      val rolesAfterDelete = roleService.readRoleAssignments().asScala

      assert(!rolesAfterDelete.exists(_.hasNameAs(developerRole)))
      assert(rolesAfterDelete.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(rolesAfterDelete.exists(_.hasNameAndPrincipalsAs(administratorRole)))
    }

    it("should remove role by id") {
      val roles = roleService.readRoleAssignments().asScala

      assert(roles.exists(_.hasNameAndPrincipalsAs(developerRole)))
      assert(roles.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(roles.exists(_.hasNameAndPrincipalsAs(administratorRole)))

      roleService.deleteById(roles.find(_.getName.equals(developerRole.getName)).get.getId)

      val rolesAfterDelete = roleService.readRoleAssignments().asScala
      assert(!rolesAfterDelete.exists(_.hasNameAs(developerRole)))
      assert(rolesAfterDelete.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(rolesAfterDelete.exists(_.hasNameAndPrincipalsAs(administratorRole)))
    }

    it("should rename role") {
      val roles = roleService.readRoleAssignments().asScala

      assert(roles.exists(_.hasNameAndPrincipalsAs(developerRole)))
      assert(roles.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(roles.exists(_.hasNameAndPrincipalsAs(administratorRole)))

      val developerNewRoleName = "newDeveloper"
      roleService.rename(developerRole.getName, developerNewRoleName)

      val rolesAfterRename = roleService.readRoleAssignments().asScala
      assert(!rolesAfterRename.exists(_.hasNameAs(developerRole)))
      assert(rolesAfterRename.exists(r => r.getName.equals(developerNewRoleName) && r.getPrincipals.containsAll(developerRole.getPrincipals)))
      assert(rolesAfterRename.exists(_.hasNameAndPrincipalsAs(deployerRole)))
      assert(rolesAfterRename.exists(_.hasNameAndPrincipalsAs(administratorRole)))

    }
  }
}
