package com.xebialabs.deployit.security.authentication;

import com.xebialabs.deployit.security.UserService;
import org.junit.Before;
import org.junit.Test;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

public abstract class AuthenticationProviderTest {

    protected static final String ANONYMOUS_USERNAME = "anonymous";

    protected static final String ADMIN_USERNAME = "admin";

    protected static final String ADMIN_PASSWORD = "admin";

    protected static final String REGULAR_USERNAME = "regular-user";

    protected static final String REGULAR_PASSWORD = "the-password-of-the-regular-user";

    private AuthenticationProvider provider;

    protected abstract UserService getUserService();

    @Before
    public void setupAuthenticationProvider() throws IllegalAccessException {
        provider = new XlAuthenticationProvider(getUserService());
    }

    @Test(expected = BadCredentialsException.class)
    public void anonymousUserShouldNotAuthenticateWithAnyPassword() {
        final String anonymousPassword = "1234";
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(ANONYMOUS_USERNAME, anonymousPassword);
        provider.authenticate(token);
    }

    @Test
    public void adminUserShouldAuthenticateWithCorrectPassword() {
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(ADMIN_USERNAME, ADMIN_PASSWORD);
        Authentication authenticated = provider.authenticate(token);
        assertThat(authenticated.getPrincipal().toString(), is(ADMIN_USERNAME));
        assertThat(authenticated.getCredentials().toString(), is(ADMIN_PASSWORD));
    }

    @Test(expected = BadCredentialsException.class)
    public void adminUserShouldNotAuthenticateWithIncorrectPassword() {
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(ADMIN_USERNAME, ADMIN_PASSWORD + "-incorrect");
        provider.authenticate(token);
    }

    @Test
    public void regularUserShouldAuthenticateWithCorrectPassword() {
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(REGULAR_USERNAME, REGULAR_PASSWORD);
        Authentication authenticated = provider.authenticate(token);
        assertThat(authenticated.getPrincipal().toString(), is(REGULAR_USERNAME));
        assertThat(authenticated.getCredentials().toString(), is(REGULAR_PASSWORD));
    }

    @Test(expected = BadCredentialsException.class)
    public void regularUserShouldNotAuthenticateWithIncorrectPassword() {
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(REGULAR_USERNAME, REGULAR_PASSWORD + "-incorrect");
        provider.authenticate(token);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldNotAuthenticateWithWrongUsername() {
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(REGULAR_USERNAME + "-incorrect", REGULAR_PASSWORD);
        provider.authenticate(token);
    }

}
