package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.security.{Role, RoleService}
import com.xebialabs.xlplatform.test.UnitTestBase

import scala.collection.JavaConverters._

trait RoleService_GlobalTest extends UnitTestBase {
  implicit val roleService: RoleService

  var roleAssignments: Seq[Role] = Seq[Role]()
  var developer: Role = _
  var deployer: Role = _
  var administrator: Role = _

  protected def setup(): Unit = {
    developer = new Role("developer")
    deployer = new Role("deployer")
    administrator = new Role("administrator")
    developer.getPrincipals.add("john")
    deployer.getPrincipals.addAll(Seq("john", "sam").asJava)
    administrator.getPrincipals.add("kate")
    roleAssignments = Seq(developer, deployer, administrator)
  }

  protected def tearDown(): Unit = {
    roleService.writeRoleAssignments(Seq[Role]().asJava)
  }

  describe("Role Service with global roles") {
    it("should not fail on no roles read") {
      val rolePrincipalsMap = roleService.readRoleAssignments.asScala

      assert(rolePrincipalsMap != null)
      assert(rolePrincipalsMap.isEmpty)
    }

    it("should read written roles") {
      val roles = roleService.getRoles.asScala

      assert(roles.isEmpty)

      roleService.writeRoleAssignments(roleAssignments.asJava)

      assert(roleService.getRoles.size() == 3)
    }

    it("should read and write role assignments") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val result = roleService.readRoleAssignments.asScala

      result should contain theSameElementsAs roleAssignments
    }

    it("should overwrite existing roles with new") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      roleService.writeRoleAssignments(List[Role]().asJava)

      assert(roleService.readRoleAssignments.size() == 0)
    }

    it("should get no roles for non existing principal") {
      val rolesFor = roleService.getRolesFor("non-existing").asScala

      assert(rolesFor.isEmpty)
    }

    it("should get roles for existing principals") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val rolesForJohn = roleService.getRolesFor("john").asScala

      assert(rolesForJohn.contains(developer))
      assert(rolesForJohn.contains(deployer))

      val rolesForKate = roleService.getRolesFor("kate").asScala

      assert(rolesForKate.contains(administrator))
    }

    it("should get role by role name") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val role = roleService.getRoleForRoleName("developer")

      role.getId should not be null
      role.getName shouldBe "developer"
    }

    it("should throw exception when fetching role by unknown role name") {
      roleService.writeRoleAssignments(roleAssignments.asJava)

      the[Exception] thrownBy roleService.getRoleForRoleName("unknown") should have('message ("Could not find the role [unknown]"))
    }

    it("should assign new id to new role") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val loser: Role = new Role("loser")
      loser.getPrincipals.add("mallory")
      val otherRoleAssignments: Seq[Role] = loser +: roleAssignments.filterNot(_.equals(developer))
      roleService.writeRoleAssignments(otherRoleAssignments.asJava)
      val roles = roleService.getRoles.asScala

      assert(!roles.contains(developer))
      assert(roles.contains(loser))
      assert(roles(roles.indexOf(loser)).getId != null)
    }

    it("should assign new id to new role with minus 1 Id") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val loser: Role = new Role(null, "loser")
      loser.getPrincipals.add("mallory")
      val otherRoleAssignments: Seq[Role] = loser +: roleAssignments.filterNot(_.equals(developer))
      roleService.writeRoleAssignments(otherRoleAssignments.asJava)
      val roles = roleService.getRoles.asScala

      assert(!roles.contains(developer))
      assert(roles.contains(loser))
      assert(roles(roles.indexOf(loser)).getId != null)
    }

    it("should not merge two roles with empty role id") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val loser: Role = new Role("loser")
      loser.getPrincipals.add("mallory")
      val superman: Role = new Role("superman")
      superman.getPrincipals.add("clark")
      val otherRoleAssignments: Seq[Role] = loser +: superman +: roleAssignments

      roleService.writeRoleAssignments(otherRoleAssignments.asJava)
      val roles = roleService.readRoleAssignments.asScala

      assert(roles.contains(loser))
      assert(roles.contains(superman))
      assert(roles(roles.indexOf(loser)).getPrincipals.contains("mallory"))
      assert(!roles(roles.indexOf(loser)).getPrincipals.contains("clark"))
      assert(roles(roles.indexOf(superman)).getPrincipals.contains("clark"))
      assert(!roles(roles.indexOf(superman)).getPrincipals.contains("mallory"))
    }

    it("should be able to update a role name") {
      roleService.writeRoleAssignments(roleAssignments.asJava)
      val savedAssignments = roleService.readRoleAssignments.asScala
      val devRole = savedAssignments.find(_.getId == developer.getId).head
      devRole.setName("ScalaDeveloper")

      roleService.writeRoleAssignments(savedAssignments.asJava)
      val updatedAssignments = roleService.readRoleAssignments.asScala
      val updatedDevRole = updatedAssignments.find(role => role.getName == devRole.getName && role.getId == devRole.getId).head

      assert(updatedAssignments.size == roleAssignments.size)
      assert(updatedDevRole.getPrincipals.containsAll(developer.getPrincipals))
    }
  }
}
