package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security.{Role, RoleService, Team, TeamService}
import com.xebialabs.xlplatform.test.ci.CiHelper
import org.scalatest.{FunSpecLike, Matchers}
import org.springframework.security.authentication.TestingAuthenticationToken

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._

trait TeamServiceTest extends FunSpecLike with Matchers {
  var roleService: RoleService = _
  var teamService: TeamService = _
  var repositoryService: RepositoryService = _

  var dirDev: Directory = _
  var env1: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _

  def setup(): Unit = {
    val ciHelper = new CiHelper(repositoryService)
    dirDev = ciHelper.createDirectory("Environments/dev")
    env1 = ciHelper.createConfigurationItem(classOf[Environment], dirDev.getId + "/env1")

    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    val aTeam = new Team("A-Team").withPrincipals("hannibal", "murdock", "face", "ba").withRoles(roleAdmin.getName)
    val bTeam = new Team("B-Team").withRoles(roleDev.getName)
    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("dev-group")

    roleService.writeRoleAssignments(Seq(roleDev, roleAdmin))
    teamService.writeTeams(Seq(aTeam, bTeam), dirDev.getId)
  }

  describe("Team service") {
    it("should get all teams from configuration item") {
      val teams = teamService.getTeams(dirDev.getId)
      teams should have size 2
      teams.get(0) should have(
        'name ("A-Team"),
        'principals (Seq("hannibal", "murdock", "face", "ba").asJava),
        'roles (Seq(roleAdmin.getName).asJava)
      )
      teams.get(1) should have(
        'name ("B-Team"),
        'principals (Seq().asJava),
        'roles (Seq(roleDev.getName).asJava)
      )

      teams.get(0).getId should not be empty
      teams.get(1).getId should not be empty
    }
    it("should get empty list when configuration item has no teams assigned") {
      val teams = teamService.getTeams(env1.getId)
      teams shouldBe empty
    }
    it("should throw exception when configuration item not found") {
      the[Exception] thrownBy teamService.getTeams("Environments/unknown") should have('message ("Configuration item [Environments/unknown] not found"))
    }
    it("should create new teams when updating item without teams") {
      val cTeam = new Team("C-Team").withPrincipals("dude", "other dude").withRoles(roleDev.getName, roleAdmin.getName)

      teamService.writeTeams(Seq(cTeam), env1.getId)

      val teams = teamService.getTeams(env1.getId)
      teams should have size 1
      teams.get(0) should have(
        'name (cTeam.getName),
        'principals (cTeam.getPrincipals),
        'roles (cTeam.getRoles)
      )
      teams.get(0).getId should not be empty
    }

    it("should throw exception when creating a team referencing an unknown role name") {
      val badTeam = new Team("Bad-Team").withRoles("unknown role")

      the[Exception] thrownBy teamService.writeTeams(Seq(badTeam), env1.getId) should have('message ("Role [unknown role] not found"))
    }
    it("should override existing teams when updating item with teams") {
      val ciId = dirDev.getId
      val teamsToUpdate = teamService.getTeams(ciId).filterNot(_.getName == "B-Team")
      val aTeam = teamsToUpdate.find(_.getName == "A-Team").get
      aTeam.getPrincipals.remove("ba")
      aTeam.getPrincipals.add("tawnia")
      aTeam.withRoles(roleAdmin.getName, roleDev.getName)

      val cTeam = new Team("C-Team").withPrincipals("dude").withRoles(roleDev.getName)

      teamService.writeTeams(teamsToUpdate :+ cTeam, ciId)

      val updatedTeams = teamService.getTeams(ciId)
      updatedTeams should have size 2
      val updatedATeam = updatedTeams.find(_.getName == "A-Team").get
      val updatedCTeam = updatedTeams.find(_.getName == "C-Team").get

      updatedATeam should have(
        'name ("A-Team"),
        'principals (Seq("hannibal", "murdock", "face", "tawnia").asJava),
        'roles (Seq(roleAdmin.getName, roleDev.getName).asJava)
      )
      updatedATeam.getId shouldBe aTeam.getId

      updatedCTeam should have(
        'name ("C-Team"),
        'principals (Seq("dude").asJava),
        'roles (Seq(roleDev.getName).asJava)
      )
      updatedCTeam.getId should not be empty
    }

    it("should get all teams matching either the principal or role") {
      val auth = new TestingAuthenticationToken("murdock", "chopper", "dev-group")
      val teams = teamService.getTeamsFor(auth, dirDev.getId)

      teams should have size 2
    }

    it("should get empty list when no matching principals or roles") {
      val auth = new TestingAuthenticationToken("unknown", "unknown", "unknown")
      val teams = teamService.getTeamsFor(auth, dirDev.getId)

      teams shouldBe empty
    }
  }
}
