package com.xebialabs.deployit.security;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;

public abstract class RoleServiceTest {

    private RoleService roleService;
    private List<Role> roleAssignments;
    private Role developer;
    private Role deployer;
    private Role administrator;

    protected abstract RoleService getRoleService();

    @Before
    public void init() {
        roleService = getRoleService();
        developer = new Role("developer");
        deployer = new Role("deployer");
        administrator = new Role("administrator");
        developer.getPrincipals().add("john");
        deployer.getPrincipals().addAll(Arrays.asList("john", "sam"));
        administrator.getPrincipals().add("kate");
        roleAssignments = Arrays.asList(developer, deployer, administrator);
    }

    @After
    public void clear() {
        roleService.writeRoleAssignments(new ArrayList<>());
    }

    @Test
    public void shouldNotFailOnNoRolesRead() {
        List<Role> rolePrincipalsMap = roleService.readRoleAssignments();
        assertThat(rolePrincipalsMap, notNullValue());
        assertThat(rolePrincipalsMap, hasSize(0));
    }

    @Test
    public void shouldReadWrittenRoles() {
        List<Role> roles = roleService.getRoles();
        assertThat(roles, hasSize(0));
        roleService.writeRoleAssignments(roleAssignments);
        roles = roleService.getRoles();
        assertThat(roles, hasSize(3));
    }

    @Test
    public void shouldReadAndWriteRoleAssignments() {
        roleService.writeRoleAssignments(roleAssignments);
        List<Role> result = roleService.readRoleAssignments();
        assertThat(new HashSet<>(result), equalTo(new HashSet<>(roleAssignments)));
    }

    @Test
    public void shouldOverwriteExistingRolesWithNew() {
        roleService.writeRoleAssignments(roleAssignments);
        roleService.writeRoleAssignments(new ArrayList<>());
        assertThat(roleService.readRoleAssignments(), hasSize(0));
    }

    @Test
    public void shouldGetNoRolesForNonExistingPrincipal() {
        List<Role> rolesFor = roleService.getRolesFor("non-existing");
        assertThat(rolesFor, hasSize(0));
    }

    @Test
    public void shouldGetRolesForExistingPrincipals() {
        roleService.writeRoleAssignments(roleAssignments);
        List<Role> rolesForJohn = roleService.getRolesFor("john");
        assertThat(rolesForJohn, hasItem(developer));
        assertThat(rolesForJohn, hasItem(deployer));
        List<Role> rolesForKate = roleService.getRolesFor("kate");
        assertThat(rolesForKate, hasItem(administrator));
    }

    @Test
    public void shouldAssignNewIdToNewRole() {
        roleService.writeRoleAssignments(roleAssignments);
        List<Role> otherRoleAssignments = new ArrayList<>();
        otherRoleAssignments.addAll(roleAssignments);
        Role loser = new Role("loser");
        loser.getPrincipals().add("mallory");
        otherRoleAssignments.add(loser);
        otherRoleAssignments.remove(developer);
        roleService.writeRoleAssignments(otherRoleAssignments);
        List<Role> roles = roleService.getRoles();
        assertThat(roles, not(hasItem(developer)));
        assertThat(roles, hasItem(loser));
        assertThat(roles.get(roles.indexOf(loser)).getId(), notNullValue());
    }

    @Test
    public void shouldAssignNewIdToNewRoleWithMinus1Id() {
        roleService.writeRoleAssignments(roleAssignments);
        List<Role> otherRoleAssignments = new ArrayList<>();
        otherRoleAssignments.addAll(roleAssignments);
        Role loser = new Role(null, "loser");
        loser.getPrincipals().add("mallory");
        otherRoleAssignments.add(loser);
        otherRoleAssignments.remove(developer);
        roleService.writeRoleAssignments(otherRoleAssignments);
        List<Role> roles = roleService.getRoles();
        assertThat(roles, not(hasItem(developer)));
        assertThat(roles, hasItem(loser));
        assertThat(roles.get(roles.indexOf(loser)).getId(), notNullValue());
    }

    @Test
    public void shouldNotMergeTwoRolesWithEmptyRoleId() {
        roleService.writeRoleAssignments(roleAssignments);
        List<Role> otherRoleAssignments = new ArrayList<>();
        otherRoleAssignments.addAll(roleAssignments);
        Role loser = new Role("loser");
        loser.getPrincipals().add("mallory");
        Role superman = new Role("superman");
        superman.getPrincipals().add("clark");
        otherRoleAssignments.add(loser);
        otherRoleAssignments.add(superman);
        roleService.writeRoleAssignments(otherRoleAssignments);
        List<Role> roles = roleService.readRoleAssignments();
        assertThat(roles, hasItem(loser));
        assertThat(roles, hasItem(superman));
        assertThat(roles.get(roles.indexOf(loser)).getPrincipals(), hasItem("mallory"));
        assertThat(roles.get(roles.indexOf(loser)).getPrincipals(), not(hasItem("clark")));
        assertThat(roles.get(roles.indexOf(superman)).getPrincipals(), hasItem("clark"));
        assertThat(roles.get(roles.indexOf(superman)).getPrincipals(), not(hasItem("mallory")));
    }

}
