package com.xebialabs.xlplatform.security

import java.util.{Set => JSet}
import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions._
import com.xebialabs.deployit.security.permission.PlatformTestPermissions.IMPLICIT_READ
import com.xebialabs.xlplatform.test.ci.CiHelper
import org.scalatest.{FunSpecLike, Matchers}

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._

trait PermissionEditorTest extends FunSpecLike with Matchers {

  var roleService: RoleService = _
  var teamService: TeamService = _
  var permissionEditor: PermissionEditor = _
  var ciHelper: CiHelper = _

  var directory1: Directory = _
  var directory11: Directory = _
  var environment111: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _

  var aTeam: Team = _
  var bTeam: Team = _

  def setup(): Unit = {
    directory1 = ciHelper.createDirectory("Environments/dev1")
    directory11 = ciHelper.createDirectory("Environments/dev1/dev11")
    environment111 = ciHelper.createEnvironment("dev1/dev11/env111")

    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    aTeam = new Team("A-Team").withPrincipals("hannibal", "murdock", "face", "ba").withRoles(roleAdmin.getName)
    bTeam = new Team("B-Team").withRoles(roleDev.getName)
    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("dev-group")

    roleService.writeRoleAssignments(Seq(roleDev, roleAdmin))
    teamService.writeTeams(Seq(aTeam, bTeam), directory1.getId)

    val permissions = Map(roleAdmin -> Set(IMPLICIT_READ).asJava, aTeam -> Set(READ, EDIT_REPO).asJava, bTeam -> Set[Permission]().asJava)
    permissionEditor.editPermissions(directory1.getId, permissions)
  }

  describe("Permission editor") {
    describe("when reading permissions") {
      it("should return empty permission map when no permissions assigned") {
        val permissions = permissionEditor.readPermissions(directory11.getId)
        permissions shouldBe empty
      }
      it("should return permission map with all associated roles and teams") {
        val permissions = permissionEditor.readPermissions(directory1.getId)
        permissions should have size 2
        permissions should contain key aTeam
        permissions(aTeam) should contain only(READ, EDIT_REPO)
        permissions should contain key roleAdmin
        permissions(roleAdmin) should contain only IMPLICIT_READ
        permissions shouldNot contain key bTeam
      }
      it("should ignore roles and teams which do not exist anymore") {
        roleService.writeRoleAssignments(Seq(roleDev))

        val permissions = permissionEditor.readPermissions(directory1.getId)

        permissions should have size 1
        permissions should contain key aTeam
        permissions shouldNot contain key roleAdmin
      }
      it("should throw exception when target configuration item not found") {
        the[Exception] thrownBy permissionEditor.readPermissions("Environments/unknowndir") should have('message ("Couldn't find a node at [Environments/unknowndir]"))

      }
      it("should throw exception when target configuration item not marked as securable") {
        the[Exception] thrownBy permissionEditor.readPermissions(environment111.getId) should have('message ("The type [udm.Environment] does not support security permissions, because it doesn't implement com.xebialabs.deployit.repository.core.Securable"))
      }
    }

    describe("when writing permissions") {
      it("should update permission map on the target configuration item") {
        val permissions = Map(roleDev -> Set(READ).asJava, bTeam -> Set(EDIT_REPO).asJava)
        permissionEditor.editPermissions(directory1.getId, permissions)

        permissions should have size 2
        permissions should contain key roleDev
        permissions(roleDev) should contain only READ
        permissions should contain key bTeam
        permissions(bTeam) should contain only EDIT_REPO
      }
      it("should throw exception when granting permission which isn't applicable to the context") {
        the[Exception] thrownBy permissionEditor.editPermissions(directory1.getId, Map(roleDev -> Set(ADMIN).asJava)) should have('message ("The permissions [admin] are not applicable to [Environments/dev1]"))
      }

      it("should throw exception when target configuration item not found") {
        the[Exception] thrownBy permissionEditor.editPermissions("Environments/unknowndir", Map[Role, JSet[Permission]]()) should have('message ("Couldn't find a node at [Environments/unknowndir]"))
      }
    }
  }
}
