package com.xebialabs.deployit.security;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.security.authentication.TestingAuthenticationToken;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PermissionHandler;
import com.xebialabs.deployit.security.permission.PlatformTestPermissions;
import com.xebialabs.xlplatform.test.ci.CiHelper;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_REPO;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.LOGIN;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;

public abstract class PermissionEnforcerTest {

    private PermissionHandler loginHandler;
    private PermissionHandler readHandler;
    private Directory envGroup;
    private Directory envDept;
    private Environment environment;
    private PermissionEnforcer permissionEnforcer;
    private Directory appGroup;
    private RoleService roleService;
    private TeamService teamService;
    private Role devver;
    private Role admin;
    private Team teamA;
    private Team teamB;

    private RepositoryService repositoryService;

    private TestPermissions permissions;

    protected abstract RoleService getRoleService();

    protected abstract TeamService getTeamService();

    protected abstract PermissionChecker getPermissionChecker();

    protected abstract PermissionEditor getPermissionEditor();

    protected abstract RepositoryService getRepositoryService();

    @Before
    public void setup() {
        roleService = getRoleService();
        teamService = getTeamService();
        PermissionEditor pe = getPermissionEditor();
        permissions = new TestPermissions(roleService, pe);
        permissionEnforcer = new PermissionEnforcer(getPermissionChecker(), roleService);
        new SecurityServiceLocator(permissionEnforcer);
        loginHandler = LOGIN.getPermissionHandler();
        readHandler = READ.getPermissionHandler();
        repositoryService = getRepositoryService();
        CiHelper ciHelper = new CiHelper(repositoryService);
        envDept = ciHelper.createDirectory("Environments/dept");
        envGroup = ciHelper.createDirectory("Environments/dept/group");
        appGroup = ciHelper.createDirectory("Applications/group");
        environment = ciHelper.createConfigurationItem(Environment.class, envGroup.getId() + "/env1");
        devver = new Role("devver");
        admin = new Role("administrator");
        teamA = new Team("A-Team").withRoles(devver.getName());
        teamB = new Team("B-Team").withPrincipals("user-1");
        devver.getPrincipals().add("group-1");
        admin.getPrincipals().add("admin-group");
        roleService.writeRoleAssignments(newArrayList(devver, admin));
        teamService.writeTeams(newArrayList(teamA), envGroup.getId());
        teamService.writeTeams(newArrayList(teamA, teamB), envDept.getId());
        // This took me some figuring out. The repository filters the securitypermissions unless you have the permission:
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("admin", "admin", "ROLE_ADMIN"));
        permissions.grant(EDIT_SECURITY, devver, null);
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "group-1"));
    }

    @After
    public void cleanup() {
        repositoryService.delete(envGroup.getId());
        repositoryService.delete(envDept.getId());
        repositoryService.delete(appGroup.getId());
        roleService.writeRoleAssignments(Lists.<Role>newArrayList());
    }

    @Test
    public void shouldCheckForPermissionGrantedToRole() {
        assertThat(loginHandler.hasPermission(null), equalTo(false));
        permissions.grant(LOGIN, devver, null);
        assertThat(loginHandler.hasPermission(null), equalTo(true));
        permissions.revoke(LOGIN, devver, null);
        assertThat(loginHandler.hasPermission(null), equalTo(false));
    }

    @Test
    public void shouldCheckForCiGroupPermissionGrantedToRole() {
        checkPermissions(devver, environment.getId(), envGroup.getId());
    }

    @Test
    public void shouldCheckForCiPermissionGrantedToTeamOnSameNode() {
        checkPermissions(teamA, envGroup.getId(), envGroup.getId());
    }

    @Test
    public void shouldCheckForCiPermissionGrantedToTeamOnParentNode() {
        checkPermissions(teamA, environment.getId(), envGroup.getId());
    }


    private void checkPermissions(final Role role, final String onConfiguration, final String ci) {
        assertThat(readHandler.hasPermission(onConfiguration), equalTo(false));
        permissions.grant(READ, role, ci);
        assertThat(readHandler.hasPermission(onConfiguration), equalTo(true));
        permissions.revoke(READ, role, ci);
        assertThat(readHandler.hasPermission(onConfiguration), equalTo(false));
    }

    @Test
    public void shouldHaveReadPermissionWhenGrantedCiLevelPermission() {
        checkRelatedReadPermission(READ, envGroup.getId(), envGroup.getId());
        checkRelatedReadPermission(PlatformTestPermissions.IMPLICIT_READ_APP, appGroup.getId(), appGroup.getId());
    }

    private void checkRelatedReadPermission(Permission p, String grantedNode, String checkNode) {
        permissions.grant(p, devver, grantedNode);
        assertThat(readHandler.hasPermission(checkNode), equalTo(true));
        assertThat(p.getPermissionHandler().hasPermission(checkNode), equalTo(true));
        permissions.revoke(p, devver, grantedNode);
        assertThat(p.getPermissionHandler().hasPermission(checkNode), equalTo(false));
        assertThat(readHandler.hasPermission(checkNode), equalTo(false));
    }

    @Test
    public void shouldNotRevokedExplicitReadWhenRevokingAnotherPermission() {
        permissions.grant(READ, devver, envGroup.getId());
        PermissionHandler deployHandler = PlatformTestPermissions.IMPLICIT_READ.getPermissionHandler();
        permissions.grant(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        assertThat(deployHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId());
        assertThat(deployHandler.hasPermission(envGroup.getId()), equalTo(false));
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(READ, devver, envGroup.getId());
    }

    @Test
    public void shouldNotRevokedExplicitReadWhenRevokingAnotherPermissionFromOtherTeam() {
        PermissionHandler editRepoHandler = EDIT_REPO.getPermissionHandler();

        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));

        permissions.grant(READ, teamA, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));

        permissions.grant(EDIT_REPO, teamB, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(true));

        permissions.revoke(EDIT_REPO, teamB, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));

        permissions.revoke(READ, teamA, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));
    }

    @Test
    public void shouldNotRemoveReadPermissionForUserWhenAnotherUserHasGrantedRelatedPermission() {
        permissions.grant(READ, devver, envGroup.getId());

        // Yes, user-2 is here on purpose...
        permissions.grant(PlatformTestPermissions.IMPLICIT_READ, admin, envGroup.getId());
        permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, admin, envGroup.getId());
        permissions.revoke(READ, devver, envGroup.getId());
    }

    @Test
    public void shouldBeAdminUserWhenGrantedAdminPermission() {
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(false));
        permissions.grant(ADMIN, devver, null);
        assertThat(permissionEnforcer.isAdmin(newArrayList("user-1"), newArrayList(devver)), equalTo(true));
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(ADMIN, devver, null);
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(false));
    }

    @Test
    public void shouldBeAdminUserWhenGrantedCorrectAuthority() {
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "ROLE_ADMIN"));
        assertThat(permissionEnforcer.isAdmin(newArrayList("user-1", "ROLE_ADMIN"), Lists.<Role>newArrayList()), equalTo(true));
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
    }

    @Test
    public void shouldRevokePermissionInOneGoWhenGrantedTwice() {
        permissions.grant(READ, devver, envGroup.getId());
        permissions.grant(READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(false));
    }

    @Test
    public void shouldRevokePermissionWhenRemovingUserFromTeamMembership() {
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));
        permissions.grant(READ, teamB, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));

        teamB.withPrincipals("unknown");
        teamService.writeTeams(newArrayList(teamB), envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));

        teamB.withPrincipals("user-1");
        teamService.writeTeams(newArrayList(teamB), envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
    }
}
