package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions._
import com.xebialabs.deployit.security.permission.PlatformTestPermissions._
import com.xebialabs.xlplatform.test.ci.CiHelper
import org.scalatest.{FunSpecLike, Matchers}
import org.springframework.security.authentication.TestingAuthenticationToken

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._


trait PermissionListerTest extends FunSpecLike with Matchers {

  var roleService: RoleService = _
  var teamService: TeamService = _
  var permissionEditor: PermissionEditor = _
  var ciHelper: CiHelper = _
  var permissionLister: PermissionLister = _
  var repositoryService: RepositoryService = _

  var directory1: Directory = _
  var directory11: Directory = _
  var directory111: Directory = _
  var environment111: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _
  var roleEmpty: Role = _
  var roleUnknown: Role = _

  var aTeam: Team = _
  var bTeam: Team = _

  def setup(): Unit = {
    directory1 = ciHelper.createDirectory("Environments/dev1")
    directory11 = ciHelper.createDirectory("Environments/dev1/dev11")
    directory111 = ciHelper.createDirectory("Environments/dev1/dev111")
    environment111 = ciHelper.createEnvironment("dev1/dev11/env111")

    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    aTeam = new Team("A-Team").withRoles(roleAdmin.getName)
    bTeam = new Team("B-Team").withRoles(roleDev.getName)
    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("group-1")

    roleEmpty = new Role("EMPTY")
    roleUnknown = new Role("???")

    roleService.writeRoleAssignments(Seq(roleDev, roleAdmin, roleEmpty))
    teamService.writeTeams(Seq(aTeam, bTeam), directory1.getId)

    val permissions = Map(roleDev -> Set(READ).asJava, roleAdmin -> Set(IMPLICIT_READ,EDIT_REPO).asJava, aTeam -> Set[Permission]().asJava, bTeam -> Set(READ, EDIT_REPO).asJava)
    permissionEditor.editPermissions(directory1.getId, permissions)

    SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "group-1"))
  }

  describe("Permission lister") {
    describe("when listing role permissions") {
      it("should list all permissions for admin role") {
        val adminPermissions = permissionLister.listPermissions(roleAdmin)
        adminPermissions(directory1.getId) should contain only (IMPLICIT_READ.getPermissionName, EDIT_REPO.getPermissionName)
      }
      it("should return empty permissions for role without permissions") {
        val emptyPermissions = permissionLister.listPermissions(roleEmpty)
        emptyPermissions.values().foreach{ per => per shouldBe empty }
      }
      it("should return all permissions for 2 roles") {
        val permissions = permissionLister.listPermissions(List(roleAdmin, roleDev))
        permissions(directory1.getId) should contain only (IMPLICIT_READ.getPermissionName, EDIT_REPO.getPermissionName, READ.getPermissionName)
      }
      it("should return empty permissions because role does not exist") {
        val emptyPermissions = permissionLister.listPermissions(roleUnknown)
        emptyPermissions.values().foreach{ per => per shouldBe empty }
      }
    }
    describe("when listing team permissions") {
      it("should list all permissions for B-Team") {
        val permissions = permissionLister.listPermissions(Permissions.getAuthentication, directory1.getId)
        permissions(directory1.getId) should contain only (READ.getPermissionName, EDIT_REPO.getPermissionName)
      }
      it("should return empty permissions for A-Team") {
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("admin", "admin", "admin"))
        val permissions = permissionLister.listPermissions(Permissions.getAuthentication, directory1.getId)
        permissions(directory1.getId) shouldBe empty
      }
      it("should return empty permissions for empty directory") {
        val permissions = permissionLister.listPermissions(Permissions.getAuthentication, directory111.getId)
        permissions(directory111.getId) shouldBe empty
      }

    }
  }

}
