package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.PlatformPermissions._
import com.xebialabs.deployit.security.permission.PlatformTestPermissions._
import com.xebialabs.xlplatform.test.ci.CiHelper
import org.scalatest.funspec.AnyFunSpecLike
import org.scalatest.matchers.should.Matchers
import org.springframework.security.authentication.TestingAuthenticationToken

import scala.jdk.CollectionConverters._

trait PermissionListerTest extends AnyFunSpecLike with Matchers {
  val GLOBAL_SECURITY_ALIAS = "global"

  var roleService: RoleService = _
  var permissionEditor: PermissionEditor = _
  var ciHelper: CiHelper = _
  var permissionLister: PermissionLister = _
  var repositoryService: RepositoryService = _

  var directory1: Directory = _
  var directory11: Directory = _
  var directory111: Directory = _
  var environment111: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _
  var roleEmpty: Role = _
  var roleUnknown: Role = _

  var emptyPermissionMapSize: Int = _
  var fullPermissionMapSize: Int = _

  def setup(): Unit = {

    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    roleEmpty = new Role("EMPTY")
    roleUnknown = new Role("???")

    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("group-1")

    roleService.writeRoleAssignments(Seq(roleDev, roleAdmin, roleEmpty).asJava)

    val permissions = Map(roleDev -> Set(READ).asJava, roleAdmin -> Set(IMPLICIT_READ, EDIT_REPO).asJava).asJava
    val permissions11 = Map(roleDev -> Set(READ).asJava, roleAdmin -> Set(EDIT_REPO).asJava).asJava
    val permissionsGlobal = Map(roleAdmin -> Set(EDIT_SECURITY).asJava).asJava
    permissionEditor.editPermissions(directory1.getId, permissions)
    permissionEditor.editPermissions(directory11.getId, permissions11)
    permissionEditor.editPermissions(GLOBAL_SECURITY_ALIAS, permissionsGlobal)

    SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "group-1"))
  }

  describe("Permission lister") {
    describe("when listing all role permissions") {
      it("should list global and local permissions for a role") {
        val adminPermissions = permissionLister.listPermissions(roleAdmin, null).asScala.toMap
        (adminPermissions - "Tests") should have size fullPermissionMapSize
        adminPermissions(directory1.getId) should contain.only(IMPLICIT_READ.getPermissionName, EDIT_REPO.getPermissionName)
        adminPermissions(directory11.getId) should contain only EDIT_REPO.getPermissionName
        adminPermissions(GLOBAL_SECURITY_ALIAS) should contain only EDIT_SECURITY.getPermissionName
      }
      it("should return all permissions for 2 roles") {
        val permissions = permissionLister.listPermissions(List(roleAdmin, roleDev).asJava, null).asScala.toMap
        (permissions - "Tests") should have size fullPermissionMapSize
        permissions(directory1.getId) should contain.only(IMPLICIT_READ.getPermissionName, EDIT_REPO.getPermissionName, READ.getPermissionName)
        permissions(directory11.getId) should contain.only(EDIT_REPO.getPermissionName, READ.getPermissionName)
      }
      it("should return empty permissions for role without permissions") {
        val emptyPermissions = permissionLister.listPermissions(roleEmpty).asScala
        emptyPermissions.values.foreach { per => per should have length 0 }
      }
      it("should return empty permissions because role does not exist") {
        val emptyPermissions = permissionLister.listPermissions(roleUnknown).asScala
        emptyPermissions.values.foreach { per => per should have length 0 }
      }
    }
    describe("when listing global role permissions") {
      it("should list only global permissions for a role") {
        val adminGlobalPermissions = permissionLister.listGlobalPermissions(Seq(roleAdmin).asJava).asScala
        adminGlobalPermissions should have size 1
        adminGlobalPermissions("global") should contain only EDIT_SECURITY.getPermissionName

        val devGlobalPermissions = permissionLister.listGlobalPermissions(Seq(roleDev).asJava).asScala
        devGlobalPermissions should have size emptyPermissionMapSize
      }
    }
  }

}
