package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security._
import com.xebialabs.xlplatform.test.UnitTestBase

import scala.jdk.CollectionConverters._

trait RoleService_LocalTest extends UnitTestBase {
  implicit val roleService: RoleService

  var dirDev: Directory = _
  var env1: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _

  var countImplemented = false

  def setup(): Unit = {
    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    val aRole = new Role("A-Role").withPrincipals("hannibal", "murdock", "face", "ba").withRoles(roleAdmin.getName)
    val bRole = new Role("B-Role").withRoles(roleDev.getName)
    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("dev-group")

    roleService.writeRoleAssignments(Seq(roleDev, roleAdmin).asJava)
    roleService.writeRoleAssignments(dirDev.getId, Seq(aRole, bRole).asJava)
  }

  describe("Role service with local roles") {
    it("should get all roles from configuration item") {
      val roles = roleService.readRoleAssignments(dirDev.getId).asScala
      roles should have size 2
      val ARole = roles.filter(_.getName == "A-Role").head
      ARole should have(
        Symbol("name") ("A-Role"),
        Symbol("roles") (Seq(roleAdmin.getName).asJava)
      )
      ARole.getPrincipals should contain theSameElementsAs Seq("hannibal", "murdock", "face", "ba")
      roles.filter(_.getName == "B-Role").head should have(
        Symbol("name") ("B-Role"),
        Symbol("principals") (Seq().asJava),
        Symbol("roles") (Seq(roleDev.getName).asJava)
      )

      roles.head.getId should not be empty
      roles(1).getId should not be empty
    }

    it("should get empty list when configuration item has no roles assigned") {
      val roles = roleService.readRoleAssignments(env1.getId)
      roles should have length 0
    }

    it("should create new roles when updating item without roles") {
      val cRole = new Role("C-Role").withPrincipals("dude", "other dude").withRoles(roleDev.getName, roleAdmin.getName)

      roleService.writeRoleAssignments(env1.getId, Seq(cRole).asJava)

      val roles = roleService.readRoleAssignments(env1.getId)
      roles should have size 1
      roles.get(0).getName should be(cRole.getName)
      roles.get(0).getPrincipals should contain theSameElementsAs cRole.getPrincipals.asScala
      roles.get(0).getRoles should contain theSameElementsAs cRole.getRoles.asScala
      roles.get(0).getId should not be empty
    }

    it("should throw exception when creating a local role referencing an unknown role name") {
      val badRole = new Role("Bad-Role").withRoles("unknown role")

      the[Exception] thrownBy roleService.writeRoleAssignments(env1.getId, Seq(badRole).asJava) should have(Symbol("message") ("Role [unknown role] not found"))
    }

    it("should override existing roles when updating item with roles") {
      val ciId = dirDev.getId
      val rolesToUpdate = roleService.readRoleAssignments(ciId).asScala.filterNot(_.getName == "B-Role")
      val aRole = rolesToUpdate.find(_.getName == "A-Role").get
      aRole.getPrincipals.remove("ba")
      aRole.getPrincipals.add("tawnia")
      aRole.withRoles(roleAdmin.getName, roleDev.getName)

      val cRole = new Role("C-Role").withPrincipals("dude").withRoles(roleDev.getName)

      roleService.writeRoleAssignments(ciId, (rolesToUpdate :+ cRole).asJava)

      val updatedRoles = roleService.readRoleAssignments(ciId).asScala
      updatedRoles should have size 2
      val updatedARole = updatedRoles.find(_.getName == "A-Role").get
      val updatedCRole = updatedRoles.find(_.getName == "C-Role").get

      updatedARole.getName should be("A-Role")
      updatedARole.getPrincipals should contain theSameElementsAs Seq("hannibal", "murdock", "face", "tawnia")
      updatedARole.getRoles should contain theSameElementsAs Seq(roleAdmin.getName, roleDev.getName)
      updatedARole.getId shouldBe aRole.getId

      updatedCRole.getName should be("C-Role")
      updatedCRole.getPrincipals should contain theSameElementsAs Seq("dude")
      updatedCRole.getRoles should contain theSameElementsAs Seq(roleDev.getName)
      updatedCRole.getId should not be empty
    }

    if (countImplemented) {
      it("should count roles by ciId") {
        assert(roleService.countRoles(dirDev.getId, "") == 3)
        assert(roleService.countRoles(dirDev.getId, "de") == 2)
      }
    }
  }
}
