package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission.definePermission
import com.xebialabs.deployit.security.permission.PermissionHandler.Level.CI
import com.xebialabs.deployit.security.permission.PlatformPermissions._
import com.xebialabs.deployit.security.permission.PlatformTestPermissions.IMPLICIT_READ
import com.xebialabs.deployit.security.permission.{Permission, ReadPermissionHandler}
import org.scalatest.funspec.AnyFunSpecLike
import org.scalatest.matchers.should.Matchers

import scala.jdk.CollectionConverters._

trait PermissionEditorTest extends AnyFunSpecLike with Matchers {

  implicit val roleService: RoleService
  implicit val permissionEditor: PermissionEditor

  var directory1: Directory = _
  var directory11: Directory = _
  var environment111: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _

  var aRole: Role = _
  var bRole: Role = _
  var cRole: Role = _
  var ccRole: Role = _
  var skipPaginationTests = true

  val DELETE: Permission = definePermission("delete", CI, new ReadPermissionHandler, true)
  val WRITE: Permission = definePermission("write", CI, new ReadPermissionHandler, true)

  def setup(): Unit = {
    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    aRole = new Role("A-Role").withPrincipals("hannibal", "murdock", "face", "ba").withRoles(roleAdmin.getName)
    bRole = new Role("B-Role").withRoles(roleDev.getName)
    cRole = new Role("C-Role").withRoles(roleDev.getName)
    ccRole = new Role("CC-Role").withRoles(roleDev.getName)
    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("dev-group")

    roleService.writeRoleAssignments(Seq(roleDev, roleAdmin).asJava)
    roleService.writeRoleAssignments(directory1.getId, Seq(aRole, bRole, cRole, ccRole).asJava)

    val permissions = Map(
      roleAdmin -> Set(IMPLICIT_READ).asJava,
      aRole -> Set(READ, EDIT_REPO).asJava,
      bRole -> Set[Permission]().asJava,
      cRole -> Set(WRITE).asJava,
      ccRole -> Set(DELETE).asJava
    )
    permissionEditor.editPermissions(directory1.getId, permissions.asJava)
  }

  describe("Permission editor") {
    describe("when reading permissions") {
      it("should return empty permission map when no permissions assigned") {
        val permissions = permissionEditor.readPermissions(directory11.getId)
        permissions.size() shouldBe 0
      }
      it("should return permission map with all associated roles and teams") {
        val permissions = permissionEditor.readPermissions(directory1.getId).asScala
        permissions should have size 4
        permissions should contain key aRole
        permissions(aRole) should contain.only(READ, EDIT_REPO)
        permissions should contain key roleAdmin
        permissions(roleAdmin) should contain only IMPLICIT_READ
        permissions shouldNot contain key bRole
      }
      it("should ignore roles and teams which do not exist anymore") {
        roleService.writeRoleAssignments(Seq(roleDev).asJava)

        val permissions = permissionEditor.readPermissions(directory1.getId)
        permissions should have size 3
        permissions should contain key aRole
        permissions shouldNot contain key roleAdmin
      }

      if (!skipPaginationTests) {
        it("should filter permissions by role name pattern") {
          val permissions = permissionEditor.readPermissions(directory1.getId, "c", null, null).asScala
          permissions should have size 2
          permissions should contain key cRole
          permissions(cRole) should contain only WRITE
          permissions should contain key ccRole
          permissions(ccRole) should contain only DELETE
          permissions shouldNot contain key bRole
        }
        it("should paginate permissions") {
          val permissions = permissionEditor.readPermissions(directory1.getId, null, new Paging(4, 1), new Ordering("asc")).asScala
          permissions should have size 1
          permissions should contain key ccRole
          permissions(ccRole) should contain only DELETE
          permissions shouldNot contain key cRole
        }
        it("should sort filtered permissions by role name") {
          val permissions = permissionEditor.readPermissions(directory1.getId, "c", new Paging(2, 1), new Ordering("asc")).asScala
          permissions should have size 1
          permissions should contain key ccRole
          permissions(ccRole) should contain only DELETE
          permissions shouldNot contain key cRole
        }
      }
    }

    describe("when writing permissions") {
      it("should update permission map on the target configuration item") {
        val permissions = Map(roleDev -> Set(READ).asJava, bRole -> Set(EDIT_REPO).asJava)
        permissionEditor.editPermissions(directory1.getId, permissions.asJava)

        permissions should have size 2
        permissions should contain key roleDev
        permissions(roleDev) should contain only READ
        permissions should contain key bRole
        permissions(bRole) should contain only EDIT_REPO
      }
      it("should throw exception when granting permission which isn't applicable to the context") {
        the[Exception] thrownBy permissionEditor.editPermissions(directory1.getId, Map(roleDev -> Set(ADMIN).asJava).asJava) should have(
          Symbol("message") (s"The permissions [admin] are not applicable to [${directory1.getId}]"))
      }
    }
  }
}
