package com.xebialabs.xlplatform.security

import com.xebialabs.deployit.plugin.api.udm.Environment
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security.{Role, RoleService}
import com.xebialabs.xlplatform.test.UnitTestBase

import scala.jdk.CollectionConverters._

trait CrudRoleService_LocalTest extends UnitTestBase {
  implicit val roleService: RoleService

  var dirDevCrud: Directory = _
  var crudEnv: Environment = _

  var roleDev: Role = _
  var roleAdmin: Role = _

  var countImplemented = false

  def setup(): Unit = {
    roleDev = new Role("ROLE_DEV")
    roleAdmin = new Role("ROLE_ADMIN")
    roleAdmin.getPrincipals.add("admin")
    roleDev.getPrincipals.add("dev-group")

    roleService.createOrUpdateRole(roleDev)
    roleService.createOrUpdateRole(roleAdmin)


    val aRole = new Role("A-Crud-Role").withPrincipals("hannibal", "murdock", "face", "ba").withRoles(roleAdmin.getName)
    val bRole = new Role("B-Crud-Role").withRoles(roleDev.getName)
    roleService.createOrUpdateRole(aRole, dirDevCrud.getId)
    roleService.createOrUpdateRole(bRole, dirDevCrud.getId)
  }

  describe("Role service with local roles") {
    it("should get all roles created for configuration item") {
      val roles = roleService.readRoleAssignments(dirDevCrud.getId).asScala

      roles should have size 2
      val ARole = roles.filter(_.getName == "A-Crud-Role").head
      ARole should have(
        Symbol("name") ("A-Crud-Role"),
        Symbol("roles") (Seq(roleAdmin.getName).asJava)
      )
      ARole.getPrincipals should contain theSameElementsAs Seq("hannibal", "murdock", "face", "ba")
      roles.filter(_.getName == "B-Crud-Role").head should have(
        Symbol("name") ("B-Crud-Role"),
        Symbol("principals") (Seq().asJava),
        Symbol("roles") (Seq(roleDev.getName).asJava)
      )

      roles.head.getId should not be empty
      roles(1).getId should not be empty
    }

    it("should create new roles when updating item without roles") {
      val cRole = new Role("C-Crud-Role").withPrincipals("dude", "other dude").withRoles(roleDev.getName, roleAdmin.getName)

      roleService.createOrUpdateRole(cRole, crudEnv.getId)

      val roles = roleService.readRoleAssignments(crudEnv.getId)
      roles should have size 1
      roles.get(0).getName should be(cRole.getName)
      roles.get(0).getPrincipals should contain theSameElementsAs cRole.getPrincipals.asScala
      roles.get(0).getRoles should contain theSameElementsAs cRole.getRoles.asScala
      roles.get(0).getId should not be empty
    }

    it("should throw exception when creating a local role referencing an unknown role name") {
      val badRole = new Role("Bad-Crud-Role").withRoles("unknown role")

      the[Exception] thrownBy roleService.createOrUpdateRole(badRole, crudEnv.getId) should have(Symbol("message") ("Role [unknown role] not found"))
    }

    it("should override existing roles when updating item with roles") {
      val ciId = dirDevCrud.getId
      val rolesToUpdate = roleService.readRoleAssignments(ciId).asScala.filterNot(_.getName == "B-Crud-Role")
      val aRole = rolesToUpdate.find(_.getName == "A-Crud-Role").get
      aRole.getPrincipals.remove("ba")
      aRole.getPrincipals.add("tawnia")
      aRole.withRoles(roleAdmin.getName, roleDev.getName)

      val cRole = new Role("C-Crud-Role").withPrincipals("dude").withRoles(roleDev.getName)

      rolesToUpdate.foreach(r => {
        roleService.createOrUpdateRole(r, ciId)
      })
      roleService.createOrUpdateRole(cRole, ciId)

      val updatedRoles = roleService.readRoleAssignments(ciId).asScala
      updatedRoles should have size 3
      val updatedARole = updatedRoles.find(_.getName == "A-Crud-Role").get
      val updatedCRole = updatedRoles.find(_.getName == "C-Crud-Role").get

      updatedARole.getName should be("A-Crud-Role")
      updatedARole.getPrincipals should contain theSameElementsAs Seq("hannibal", "murdock", "face", "tawnia")
      updatedARole.getRoles should contain theSameElementsAs Seq(roleAdmin.getName, roleDev.getName)
      updatedARole.getId shouldBe aRole.getId

      updatedCRole.getName should be("C-Crud-Role")
      updatedCRole.getPrincipals should contain theSameElementsAs Seq("dude")
      updatedCRole.getRoles should contain theSameElementsAs Seq(roleDev.getName)
      updatedCRole.getId should not be empty
    }

    if (countImplemented) {
      it("should count roles by ciId") {
        assert(roleService.countRoles(dirDevCrud.getId, "") == 3)
        assert(roleService.countRoles(dirDevCrud.getId, "de") == 2)
      }
    }
  }
}
