package com.xebialabs.deployit.security;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.security.authentication.TestingAuthenticationToken;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PermissionHandler;
import com.xebialabs.deployit.security.permission.PlatformTestPermissions;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_REPO;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.LOGIN;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;

public abstract class PermissionEnforcerTest {

    private PermissionHandler loginHandler;
    private PermissionHandler readHandler;
    private PermissionEnforcer permissionEnforcer;
    private RoleService roleService;

    protected Directory envGroup;
    protected Directory envDept;
    protected Environment environment;
    protected Directory appGroup;

    private Role devver;
    private Role admin;
    private Role lRoleA;
    private Role lRoleB;

    private TestPermissions permissions;

    protected abstract RoleService getRoleService();

    protected abstract PermissionChecker getPermissionChecker();

    protected abstract PermissionEditor getPermissionEditor();

    @Before
    public void setup() {
        roleService = getRoleService();
        PermissionEditor pe = getPermissionEditor();
        permissions = new TestPermissions(roleService, pe);
        permissionEnforcer = new PermissionEnforcer(getPermissionChecker(), roleService);
        new SecurityServiceLocator(permissionEnforcer, null);
        loginHandler = LOGIN.getPermissionHandler();
        readHandler = READ.getPermissionHandler();

        devver = new Role("devver");
        admin = new Role("administrator");
        lRoleA = new Role("A-Role").withRoles(devver.getName());
        lRoleB = new Role("B-Role").withPrincipals("user-1");
        devver.getPrincipals().add("group-1");
        admin.getPrincipals().add("admin-group");
        roleService.writeRoleAssignments(newArrayList(devver, admin));
        roleService.writeRoleAssignments(envGroup.getId(), newArrayList(lRoleA));
        roleService.writeRoleAssignments(envDept.getId(), newArrayList(lRoleA, lRoleB));
        // This took me some figuring out. The repository filters the securitypermissions unless you have the permission:
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("admin", "admin", "ROLE_ADMIN"));
        permissions.grant(EDIT_SECURITY, devver, null);
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "group-1"));
    }

    @After
    public void cleanup() {
        roleService.writeRoleAssignments(Lists.newArrayList());
    }

    @Test
    public void shouldCheckForPermissionGrantedToRole() {
        assertThat(loginHandler.hasPermission((String) null), equalTo(false));
        permissions.grant(LOGIN, devver, null);
        assertThat(loginHandler.hasPermission((String) null), equalTo(true));
        permissions.revoke(LOGIN, devver, null);
        assertThat(loginHandler.hasPermission((String) null), equalTo(false));
    }

    @Test
    public void shouldCheckForCiGroupPermissionGrantedToRole() {
        checkPermissions(devver, environment.getId(), envGroup.getId());
    }

    @Test
    public void shouldCheckForCiPermissionGrantedToRoleOnSameNode() {
        checkPermissions(lRoleA, envGroup.getId(), envGroup.getId());
    }

    @Test
    public void shouldCheckForCiPermissionGrantedToRoleOnParentNode() {
        checkPermissions(lRoleA, environment.getId(), envGroup.getId());
    }

    private void checkPermissions(final Role role, final String onConfiguration, final String ci) {
        assertThat(readHandler.hasPermission(onConfiguration), equalTo(false));
        permissions.grant(READ, role, ci);
        assertThat(readHandler.hasPermission(onConfiguration), equalTo(true));
        permissions.revoke(READ, role, ci);
        assertThat(readHandler.hasPermission(onConfiguration), equalTo(false));
    }

    @Test
    public void shouldHaveReadPermissionWhenGrantedCiLevelPermission() {
        checkRelatedReadPermission(READ, envGroup.getId(), envGroup.getId());
        checkRelatedReadPermission(PlatformTestPermissions.IMPLICIT_READ_APP, appGroup.getId(), appGroup.getId());
    }

    private void checkRelatedReadPermission(Permission p, String grantedNode, String checkNode) {
        permissions.grant(p, devver, grantedNode);
        assertThat(readHandler.hasPermission(checkNode), equalTo(true));
        assertThat(p.getPermissionHandler().hasPermission(checkNode), equalTo(true));
        permissions.revoke(p, devver, grantedNode);
        assertThat(p.getPermissionHandler().hasPermission(checkNode), equalTo(false));
        assertThat(readHandler.hasPermission(checkNode), equalTo(false));
    }

    @Test
    public void shouldNotRevokedExplicitReadWhenRevokingAnotherPermission() {
        permissions.grant(READ, devver, envGroup.getId());
        PermissionHandler deployHandler = PlatformTestPermissions.IMPLICIT_READ.getPermissionHandler();
        permissions.grant(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        assertThat(deployHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId());
        assertThat(deployHandler.hasPermission(envGroup.getId()), equalTo(false));
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(READ, devver, envGroup.getId());
    }

    @Test
    public void shouldNotRevokedExplicitReadWhenRevokingAnotherPermissionFromOtherRole() {
        PermissionHandler editRepoHandler = EDIT_REPO.getPermissionHandler();

        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));

        permissions.grant(READ, lRoleA, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));

        permissions.grant(EDIT_REPO, lRoleB, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(true));

        permissions.revoke(EDIT_REPO, lRoleB, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));

        permissions.revoke(READ, lRoleA, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));
        assertThat(editRepoHandler.hasPermission(environment.getId()), equalTo(false));
    }

    @Test
    public void shouldNotRemoveReadPermissionForUserWhenAnotherUserHasGrantedRelatedPermission() {
        permissions.grant(READ, devver, envGroup.getId());

        // Yes, user-2 is here on purpose...
        permissions.grant(PlatformTestPermissions.IMPLICIT_READ, admin, envGroup.getId());
        permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(PlatformTestPermissions.IMPLICIT_READ, admin, envGroup.getId());
        permissions.revoke(READ, devver, envGroup.getId());
    }

    @Test
    public void shouldBeAdminUserWhenGrantedAdminPermission() {
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(false));
        permissions.grant(ADMIN, devver, null);
        assertThat(permissionEnforcer.isAdmin(newArrayList("user-1"), newArrayList(devver)), equalTo(true));
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(ADMIN, devver, null);
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(false));
    }

    @Test
    public void shouldBeAdminUserWhenGrantedCorrectAuthority() {
        SecurityTemplate.setAuthentication(new TestingAuthenticationToken("user-1", "user-1", "ROLE_ADMIN"));
        assertThat(permissionEnforcer.isAdmin(newArrayList("user-1", "ROLE_ADMIN"), Lists.<Role>newArrayList()), equalTo(true));
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
    }

    @Test
    public void shouldRevokePermissionInOneGoWhenGrantedTwice() {
        permissions.grant(READ, devver, envGroup.getId());
        permissions.grant(READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(true));
        permissions.revoke(READ, devver, envGroup.getId());
        assertThat(readHandler.hasPermission(envGroup.getId()), equalTo(false));
    }

    @Test
    public void shouldRevokePermissionWhenRemovingUserFromLocalRoleMembership() {
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));
        permissions.grant(READ, lRoleB, envDept.getId());
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));

        lRoleB.withPrincipals("unknown");
        roleService.writeRoleAssignments(envDept.getId(), newArrayList(lRoleB));
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(false));

        lRoleB.withPrincipals("user-1");
        roleService.writeRoleAssignments(envDept.getId(), newArrayList(lRoleB));
        assertThat(readHandler.hasPermission(environment.getId()), equalTo(true));
    }
}
