package com.xebialabs.xlplatform.security.sql

import java.util.{List => JList}

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.authentication.{AuthenticationFailureException, UserAlreadyExistsException}
import com.xebialabs.deployit.security.{RepoUser, SHA256PasswordEncoder, UserService}
import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlplatform.security.sql.db.Tables.{User, users}
import grizzled.slf4j.Logging
import slick.dbio.DBIOAction.seq
import slick.jdbc.JdbcProfile

class SqlUserService(securityDatabase: Database) extends UserService with Logging {

  import securityDatabase._

  val profile: JdbcProfile = config.databaseType.profile

  import profile.api._

  private val passwordEncoder = new SHA256PasswordEncoder()

  // for backwards compatibility with JCR
  private val JCR_ADMIN_USER = "admin"

  private def checkValidUsername(username: String): Unit = {
    if (username == null || username.length == 0) throw new IllegalArgumentException("Username can neither be null nor empty.")
    readUser(username) match {
      case Some(_) => throw new UserAlreadyExistsException(username)
      case None =>
    }
  }

  private def readUser(username: String): Option[User] =
    runAwait(
      users
        .filter(_.username.toLowerCase === username.toLowerCase)
        .map(user => (user.username, user.password)).result
    ).headOption

  override def create(username: String, password: String): Unit = {
    checkValidUsername(username)
    val pwd = passwordEncoder.encode(password)
    runAwait(seq(users += (username, pwd)))
  }

  override def read(username: String): RepoUser = {
    readUser(username)
      .map(user => new RepoUser(user._1, JCR_ADMIN_USER.equals(user._1)))
      .getOrElse(throw new NotFoundException(s"No such user: $username"))
  }

  override def listUsernames(): JList[String] = runAwait(users.map(_.username).result).toList.asJavaMutable()

  override def modifyPassword(username: String, newPassword: String): Unit =
    runAwait(users.filter(_.username === username).map(_.password).update(passwordEncoder.encode(newPassword)))

  /**
    * @throws IllegalArgumentException when the provided old password does not match
    */
  override def modifyPassword(username: String, newPassword: String, oldPassword: String): Unit = {
    readUser(username).foreach(user => {
      if (!passwordEncoder.matches(oldPassword, user._2)) throw new IllegalArgumentException("Failed to change password: Old password does not match.")
      modifyPassword(username, newPassword)
    })
  }

  override def delete(username: String): Unit = runAwait(users.filter(_.username === username).delete)

  override def authenticate(username: String, password: String): Unit = {
    val user = readUser(username).getOrElse(throw new AuthenticationFailureException(s"Cannot authenticate $username"))
    if (!passwordEncoder.matches(password, user._2)) throw new AuthenticationFailureException(s"Wrong credentials supplied for user $username")
  }
}
