package com.xebialabs.xlplatform.security

import java.util
import java.util.Locale
import javax.jcr.{Node, Session}

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.jcr._
import com.xebialabs.deployit.security.authentication.{AuthenticationFailureException, UserAlreadyExistsException}
import com.xebialabs.deployit.security.{RepoUser, User, UserService}
import com.xebialabs.xlplatform.security.ModeshapeUserService._
import org.apache.jackrabbit.core.security.user.PasswordUtility
import org.modeshape.common.text.Jsr283Encoder
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConversions._
import scala.util.{Failure, Success, Try}

object ModeshapeUserService {

  private val encoder = new Jsr283Encoder()

  val USERS_ROOT_NAME = "$users"
  val PASSWORD_PROPERTY_NAME = "password"

  /**
   * Encodes username
   */
  def encodeUsername(username: String) = encoder.encode(username.toLowerCase(Locale.ENGLISH))

  /**
   * Generates password hash using utils copied from Jackrabbit
   */
  def hashPassword(password: String) =
    PasswordUtility.buildPasswordHash(password, PasswordUtility.DEFAULT_ALGORITHM, PasswordUtility.DEFAULT_SALT_SIZE, PasswordUtility.DEFAULT_ITERATIONS)

  /**
   * Returns an absolute path to the node where user's information is stored
   */
  private def nodeOfUser(username: String)(implicit session: Session) =
    session.getNode(s"/$USERS_ROOT_NAME/${encodeUsername(username)}")

}

class ModeshapeUserService (jcrTemplate: ScalaJcrTemplate) extends UserService {

  private val logger: Logger = LoggerFactory.getLogger(classOf[ModeshapeUserService])

  def init(adminPassword: String): ModeshapeUserService = {
    jcrTemplate.execute { session =>
      if (!session.getRootNode.hasNode(USERS_ROOT_NAME)) {
        logger.debug("Initializing users root node")
        session.getRootNode.addNode(USERS_ROOT_NAME)
        session.save()
        create("admin", adminPassword)
      }
    }
    this
  }

  override def create(username: String, password: String): Unit = {

    val userNodeName = encodeUsername(username)

    jcrTemplate.execute { session =>
      val usersNode = session.getNode(s"/$USERS_ROOT_NAME")
      if (usersNode.hasNode(userNodeName)) {
        throw new UserAlreadyExistsException(username)
      }

      val userNode = usersNode.addNode(userNodeName)
      userNode.setProperty(PASSWORD_PROPERTY_NAME, hashPassword(password))

      session.save()
    }
  }

  override def modifyPassword(username: String, newPassword: String): Unit = {
    updateUserNode(username) { node =>
      node.setProperty(PASSWORD_PROPERTY_NAME, hashPassword(newPassword))
    }
  }

  override def modifyPassword(username: String, newPassword: String, oldPassword: String): Unit = {

    try {
      authenticate(username, oldPassword)
    } catch {
      case e: AuthenticationFailureException => throw new IllegalArgumentException(
        s"Failed to modify password of [$username] because old password doesn't match")
    }

    modifyPassword(username, newPassword)
  }

  override def delete(username: String): Unit = {
    updateUserNode(username) { node =>
      node.remove()
    }
  }

  override def read(username: String): User = {
    withUserNode(username) { userNode =>
      new RepoUser(username, username == JcrConstants.ADMIN_USERNAME)
    }
  }

  override def listUsernames(): util.List[String] = {
    jcrTemplate.execute { session =>
      val usersIterator = session.getNode(s"/$USERS_ROOT_NAME").getNodes

      var users: List[String] = List()
      while (usersIterator.hasNext) {
        users = users :+ usersIterator.nextNode().getName
      }

      users
    }
  }

  @throws[AuthenticationFailureException]
  override def authenticate(username: String, password: String): Unit = {
    logger.trace("Authenticating [{}]", username)
    withUserNode(username) { userNode =>
      val passwordHash = userNode.getProperty(PASSWORD_PROPERTY_NAME).getString

      if (!PasswordUtility.isSame(passwordHash, password)) {
        throw new AuthenticationFailureException(s"Wrong credentials supplied for user [$username]")
      }
    }
  }


  private def withUserNode[T](username: String)(callback: Node => T): T = {
    withUserNodeAndSession(username) { (node, session) =>
      callback(node)
    }
  }

  private def updateUserNode[T](username: String)(callback: Node => T): T = {
    withUserNodeAndSession(username) { (node, session) =>
      val result = callback(node)
      session.save()
      result
    }
  }

  private def withUserNodeAndSession[T](username: String)(callback: (Node, Session) => T): T = {
    jcrTemplate.execute { session =>
      Try(nodeOfUser(username)(session)) match {
        case Success(userNode: Node) =>
          callback(userNode, session)
        case Failure(exception) =>
          throw new NotFoundException(username)
      }
    }
  }
}
