package com.xebialabs.deployit.security;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.jcr.Credentials;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import com.xebialabs.deployit.engine.api.dto.Ordering;
import com.xebialabs.deployit.engine.api.dto.Paging;
import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.user.*;
import org.apache.jackrabbit.api.security.user.User; // otherwise we get the interface from xl-security-api
import org.apache.jackrabbit.core.security.authentication.CryptedSimpleCredentials;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException;
import com.xebialabs.deployit.security.authentication.UserAlreadyExistsException;

import static com.xebialabs.overthere.util.OverthereUtils.checkState;

public class JackrabbitUserService implements UserService {

    private final JcrTemplate jcrTemplate;

    public JackrabbitUserService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
    public long countUsers(String username) {
        return -1;
    }

    @Override
    public void create(final String username, final String password) {
        jcrTemplate.execute(session -> {
            UserManager um = ((JackrabbitSession) session).getUserManager();

            try {
                um.createUser(username, password);
            } catch (AuthorizableExistsException e) {
                throw new UserAlreadyExistsException(username);
            }
            return null;
        });
    }

    @Override
    public RepoUser read(final String username) {
        return jcrTemplate.execute(session -> {
            User u = getUser(username, session);
            return new RepoUser(u.getID(), u.isAdmin());
        });

    }

    @Override
    public List<String> listUsernames() {
        return this.listUsernames(null, null, null);
    }

    @Override
    public List<String> listUsernames(final String username, final Paging paging, final Ordering order) {
        return jcrTemplate.execute(session -> {
            UserManager um = ((JackrabbitSession) session).getUserManager();
            Iterator<Authorizable> authorizables = um.findAuthorizables(new Query() {
                @Override
                public <T> void build(QueryBuilder<T> builder) {
                    builder.setSelector(User.class);
                }
            });
            List<String> names = new ArrayList<>();
            while (authorizables.hasNext()) {
                String name = authorizables.next().getPrincipal().getName();
                if ("anonymous".equals(name)) continue;
                names.add(name);
            }
            return names;
        });
    }

    @Override
    public void modifyPassword(final String username, final String newPassword) {
        jcrTemplate.execute(session -> {
            User u = getUser(username, session);
            u.changePassword(newPassword);
            return null;
        });

    }

    @Override
    public void modifyPassword(final String username, final String newPassword, final String oldPassword) {
        jcrTemplate.execute(session -> {
            User u = getUser(username, session);
            try {
                u.changePassword(newPassword, oldPassword);
            } catch (RepositoryException e) {
                // Assuming that the problem is a non-matching password
                throw new IllegalArgumentException(e);
            }
            return null;
        });

    }

    @Override
    public void delete(final String username) {
        jcrTemplate.execute(session -> {
            User u = getUser(username, session);
            u.remove();
            return null;
        });
    }

    @Override
    public void authenticate(final String username, final String password) throws AuthenticationFailureException {
        if (JcrConstants.JCR_ADMIN_USERNAME.equals(username.toLowerCase())) {
            throw new AuthenticationFailureException("Cannot authenticate [%s]", username);
        }
        jcrTemplate.execute(session -> {
            UserManager um = ((JackrabbitSession) session).getUserManager();
            User user = (User) um.getAuthorizable(username);
            if (user == null) {
                throw new AuthenticationFailureException("Cannot authenticate [%s],", username);
            }
            Credentials credentials = user.getCredentials();
            validateCredentials(credentials, username, password);
            return null;
        });
    }

    private User getUser(final String username, Session session) throws RepositoryException {
        UserManager um = ((JackrabbitSession) session).getUserManager();

        Authorizable authorizable = um.getAuthorizable(username);
        if (authorizable == null) {
            throw new NotFoundException("No such user: " + username);
        }

        if (authorizable.isGroup()) {
            throw new IllegalStateException("Principal " + username + " is not a user");
        }

        return (User) authorizable;
    }


    private void validateCredentials(Credentials credentials, String username, String password) {
        checkState(credentials instanceof CryptedSimpleCredentials, "Should have an instance of CryptedSimpleCredentials");
        try {
            if (!((CryptedSimpleCredentials) credentials).matches(new SimpleCredentials(username, password.toCharArray()))) {
                throw new AuthenticationFailureException("Wrong credentials supplied for user [%s]", username);
            }
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
            throw new AuthenticationFailureException(e);
        }
    }

}
