package com.xebialabs.deployit.security.authentication;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.List;

import javax.jcr.Credentials;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.api.security.user.UserManager;
import org.apache.jackrabbit.core.security.authentication.CryptedSimpleCredentials;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.authority.mapping.GrantedAuthoritiesMapper;
import org.springframework.security.core.authority.mapping.NullAuthoritiesMapper;

import com.google.common.base.Strings;

import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.security.PermissionEnforcer;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;

public class JcrAuthenticationProvider implements AuthenticationProvider {

    @Autowired
    private JcrTemplate jcrTemplate;

    private GrantedAuthoritiesMapper authoritiesMapper = new NullAuthoritiesMapper();

    @Override
    public Authentication authenticate(final Authentication token) throws AuthenticationException {
        logger.debug("Authenticating for XL Deploy: {}", token.getPrincipal());
        final String username = token.getPrincipal().toString();
        final String password = token.getCredentials().toString();

        return doAuthentication(username, password);
    }

    private Authentication doAuthentication(final String username, final String password) {
        if (Strings.nullToEmpty(username).trim().isEmpty()) {
            throw new BadCredentialsException("Cannot authenticate with empty username");
        }

        try {
            logger.trace("Authenticating [{}]", username);
            return jcrTemplate.execute(new JcrCallback<Authentication>() {
                @Override
                public Authentication doInJcr(Session session) throws RepositoryException {
                    UserManager um = ((JackrabbitSession) session).getUserManager();
                    User authorizable = (User) um.getAuthorizable(username);
                    if (authorizable == null) {
                        throw new AuthenticationFailureException("User with username [%s] does not exist,", username);
                    }
                    Credentials credentials = authorizable.getCredentials();
                    validateCredentials(credentials, username, password);

                    List<GrantedAuthority> authorities = Lists.newArrayList();
                    if (authorizable.isAdmin()) {
                        authorities.add(new SimpleGrantedAuthority(PermissionEnforcer.ROLE_ADMIN));
                    }

                    Collection<? extends GrantedAuthority> mappedAuthorities = authoritiesMapper.mapAuthorities(authorities);

                    UsernamePasswordAuthenticationToken auth = new UsernamePasswordAuthenticationToken(username, password, mappedAuthorities);
                    return auth;
                }
            });
        } catch (AuthenticationFailureException exc) {
            throw new BadCredentialsException("Cannot authenticate " + username, exc);
        }
    }

    private void validateCredentials(Credentials credentials, String username, String password) {
        Preconditions.checkState(credentials instanceof CryptedSimpleCredentials, "Should have an instance of CryptedSimpleCredentials");
        try {
            if (!((CryptedSimpleCredentials) credentials).matches(new SimpleCredentials(username, password.toCharArray()))) {
                throw new AuthenticationFailureException("Wrong credentials supplied for user [%s]", username);
            }
        } catch (NoSuchAlgorithmException e) {
            throw new AuthenticationFailureException(e);
        } catch (UnsupportedEncodingException e) {
            throw new AuthenticationFailureException(e);
        }
    }

    @Override
    public boolean supports(final Class<? extends Object> authentication) {
        return authentication.isAssignableFrom(UsernamePasswordAuthenticationToken.class);
    }

    public GrantedAuthoritiesMapper getAuthoritiesMapper() {
        return authoritiesMapper;
    }

    public void setAuthoritiesMapper(GrantedAuthoritiesMapper authoritiesMapper) {
        this.authoritiesMapper = authoritiesMapper;
    }

    private static final Logger logger = LoggerFactory.getLogger(JcrAuthenticationProvider.class);
}
