package com.xebialabs.deployit.security.validator

import java.util
import java.util.regex.Pattern
import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument, checkNotNull}
import com.xebialabs.deployit.engine.api.security.User
import com.xebialabs.xlplatform.security.dto.PasswordValidationResult
import com.xebialabs.xlplatform.security.validator.{Invalid, StringValidator, Valid}
import org.apache.commons.lang.StringUtils.isBlank
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
class UserValidator @Autowired()(val passwordValidator: StringValidator) {
  def check(user: User): Unit = {
    checkNotNull(user, "User can not be null.", "")
    checkUserName(user)
    checkPassword(user)
  }

  def checkUserName(user: User): Unit = {
    checkArgument(!isBlank(user.getUsername), "User name cannot be empty.")
      val pattern = Pattern.compile("^(?![\\.\\-_])(?!.*[\\.\\-_]{2})[\\w\\.\\-_]*[^\\W_]$", Pattern.UNICODE_CHARACTER_CLASS)
      val matcher = pattern.matcher(user.getUsername)
      checkArgument(matcher.matches, "All the alphanumeric characters and the special characters dash (-), underscore (_), and dot (.) are allowed. The special characters cannot be repeated more than once in a row. The username must not start or end with a special character.")
  }

  def checkPassword(user: User): Unit = {
    checkArgument(!isBlank(user.getPassword), "Password is required when creating a user.")
    checkPassword(user.getPassword)
  }

  def checkPassword(password: String): Unit = {
    val invalids = passwordValidator.validate(password).filter(_.isInstanceOf[Invalid])
    if (invalids.nonEmpty) {
      throw new IncorrectArgumentException((List("Password needs to meet following requirements:") ++ invalids.map(_.message)).mkString(System.lineSeparator + "\t"))
    }
  }

  def validatePassword(password: String): util.List[PasswordValidationResult] = passwordValidator
    .validate(password)
    .map(vr => new PasswordValidationResult(vr.isInstanceOf[Valid], vr.message))
    .sortBy(_.isValid)(Ordering.Boolean.reverse)
    .asJava

  def checkMaxLength(msg: String, field: String): Unit = {
    checkArgument(msg.length <= 255, s"Maximum allowed length for the $field is 255 characters. Provide an entry with fewer characters.")
  }

  def checkIsValidEmail(email: String): Unit = {
    val  emailPattern = "^[A-Za-z0-9_%+-]+(?:\\.[A-Za-z0-9_%+-]+)?@[A-Za-z0-9-]+(\\.[A-Za-z0-9-]+)*(\\.[A-Za-z]{2,6})$".r
    checkArgument(emailPattern.matches(email), s"The email address $email is invalid. Provide a valid email address.")
  }
}

