package com.xebialabs.deployit.security

import java.util

import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import org.springframework.security.core.Authentication

/**
  * Reads roles from the repository and allows editing. The ordinary {@code getSomething()} methods return Roles <i>without</i> the assigned principals.
  * Only {@link #readRoleAssignments(String, Paging, Ordering)} will a return a list with Role objects that also contain the principals.
  * <p>
  * To edit the roles, first get a local copy of everything using {@link #readRoleAssignments(String, Paging, Ordering)} ()},
  * then write them all back with {@link #writeRoleAssignments(List)}.
  */
trait RoleService {
  /**
    * Checks whether the role exists already
    */
  def roleExists(roleName: String): Boolean

  //
  // Simple getters
  /**
    * Returns the total of roles for specified CI ID
    */
  def countRoles(onConfigurationItem: Number, rolePattern: String): Long

  /**
    * Returns the total of roles for specified CI path
    */
  def countRoles(onConfigurationItem: String, rolePattern: String): Long

  /**
    * Returns the global roles from the repository.
    */
  def getRoles(): util.List[Role] = this.getRoles(null, null, null)

  def getRoles(rolePattern: String, paging: Paging, order: Ordering): util.List[Role]

  /**
    * Returns the global roles associated with the passed principal.
    */
  def getRolesFor(principal: String): util.List[Role] = this.getRolesFor(principal, null, null, null)

  def getRolesFor(principal: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role]

  /**
    * Returns the global roles associated with the passed authentication.
    */
  def getRolesFor(auth: Authentication): util.List[Role] = this.getRolesFor(auth, null, null, null)

  def getRolesFor(auth: Authentication, rolePattern: String, paging: Paging, order: Ordering): util.List[Role]

  /**
    * Returns the local roles present on the passed configuration item.
    */
  def getRoles(onConfigurationItem: String): util.List[Role] = this.getRoles(onConfigurationItem, null, null, null)

  def getRoles(onConfigurationItem: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role]

  /**
    * Resolves the role name to a role (either global or local).
    */
  def getRoleForRoleName(roleName: String): Role

  /**
    * Returns the global roles with member principals and roles loaded.
    */
  def readRoleAssignments(): util.List[Role] = this.readRoleAssignments(null, null, null)

  def readRoleAssignments(rolePattern: String, paging: Paging, order: Ordering): util.List[Role]

  /**
    * Returns the local roles present on the passed configuration item with member principals and roles loaded.
    */
  def readRoleAssignments(onConfigurationItem: String): util.List[Role] = this.readRoleAssignments(onConfigurationItem, null, null, null)

  def readRoleAssignments(onConfigurationItem: String, rolePattern: String, paging: Paging, order: Ordering): util.List[Role]

  // Edit methods
  def create(name: String, onConfigurationItem: String): String

  def create(name: String): String = this.create(name, null)

  def createOrUpdateRole(role: Role, onConfigurationItem: String): String

  def createOrUpdateRole(role: Role): String = this.createOrUpdateRole(role, null)

  def rename(name: String, newName: String, onConfigurationItem: String): String

  def rename(name: String, newName: String): String = this.rename(name, newName, null)

  def deleteByName(name: String): Unit

  def deleteById(roleId: String): Unit

  /**
    * Persists the passed roles as global roles in the system.
    */
  def writeRoleAssignments(roles: util.List[Role]): Unit

  /**
    * Persists the passed roles as local roles specific to the passed configuration item.
    */
  def writeRoleAssignments(onConfigurationItem: String, roles: util.List[Role]): Unit

}
