package com.xebialabs.platform.script.jython

import java.io.{IOException, Writer}

import grizzled.slf4j.Logging

import scala.collection.mutable

/**
 * Since there is an unexpected behaviour when running multiple script engines in parallel : http://bugs.jython.org/issue2154
 * We have to resort to a StringWriter sub-class which delegates to a StringWriter found using a ThreadLocal container.
 *
 * MW=20170712: The issue referenced above is marked resolved, but our tests still indicate the problem is not fixed.
 */
class ThreadLocalWriterDecorator() extends Writer with Logging {

  val threadLocal: ThreadLocal[mutable.ArrayStack[Writer]] = ThreadLocal.withInitial(() => new mutable.ArrayStack[Writer])

  def registerWriter(writer: Writer): Unit = threadLocal.get.push(writer)

  def removeWriter(): Unit = {
    if (threadLocal.get.isEmpty) {
      logger.error(s"Tried to remove writer, but stack is empty")
    } else {
      threadLocal.get.pop
    }
  }

  def getWriter: Writer = threadLocal.get.headOption.orNull

  @throws(classOf[IOException])
  def write(cbuf: Array[Char], off: Int, len: Int): Unit = {
    val writer = getWriter
    if (writer == null) {
      logger.error(s"no writer registered for thread, logging here instead: ${String.copyValueOf(cbuf, off, len)}")
    } else {
      writer.write(cbuf, off, len)
    }
  }

  @throws(classOf[IOException])
  def flush(): Unit = {
    val writer = getWriter
    if (writer == null) {
      logger.error(s"no writer registered but flush() is called")
    } else {
      getWriter.flush()
    }
  }

  @throws(classOf[IOException])
  def close(): Unit = {
    val writer = getWriter
    if (writer == null) {
      logger.error(s"no writer registered but close() is called")
    } else {
      getWriter.close()
    }
  }

  override def toString: String = {
    val writer = getWriter
    if (writer == null) {
      logger.error(s"no writer registered but toString() is called")
      "null"
    } else {
      getWriter.toString
    }
  }
}
