package com.xebialabs.deployit.service.externalproperties

import java.util
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class ExternalPropertiesResolverImpl(@Autowired(required = false) providers: Array[ExternalValueProvider]) extends ExternalPropertiesResolver with Logging {

  override final def resolveExternalProperties(cis: ConfigurationItem*): Unit = resolveExternalProperties(cis, new mutable.HashSet[ConfigurationItem])

  override final def resolveExternalProperties(cis: util.Collection[_ <: ConfigurationItem]): Unit = resolveExternalProperties(cis.asScala.toSeq, new mutable.HashSet[ConfigurationItem])

  private def resolveExternalProperties(cis: Seq[_ <: ConfigurationItem], handled: mutable.Set[ConfigurationItem]): Unit = {
    cis.foreach(_resolveExternalProperties(_, handled))
  }

  private def _resolveExternalProperties(ci: ConfigurationItem, handled: mutable.Set[ConfigurationItem]): Unit = {
    if (!handled.contains(ci)) {
      logger.debug(s"Resolving external properties for ${ci.getId}")
      asBaseConfigurationItem(ci)(lookupAndSetProperties)
      handled.add(ci)
      recurseResolveExternalProperties(ci, handled)
    }
  }

  private def recurseResolveExternalProperties(ci: ConfigurationItem, handled: mutable.Set[ConfigurationItem]): Unit = {
    ci.getType.getDescriptor.getPropertyDescriptors.forEach { d =>
      d.getKind match {
        case PropertyKind.CI =>
          Option(d.get(ci).asInstanceOf[ConfigurationItem]).foreach(_resolveExternalProperties(_, handled))
        case PropertyKind.SET_OF_CI =>
          d.get(ci).asInstanceOf[util.Set[ConfigurationItem]].forEach(_resolveExternalProperties(_, handled))
        case PropertyKind.LIST_OF_CI =>
          d.get(ci).asInstanceOf[util.List[ConfigurationItem]].forEach(_resolveExternalProperties(_, handled))
        case _ => // Nothing
      }
    }
  }

  private def lookupAndSetProperties(bci: BaseConfigurationItem): Unit = {
    bci.get$externalProperties().forEach { (propertyName, externalProperty) =>
      val provider = providers.find(_.supports(externalProperty))
        .getOrElse(throw new IllegalArgumentException(s"No external value provider can handle $externalProperty."))
      val propertyDescriptor = bci.getType.getDescriptor.getPropertyDescriptor(propertyName)
      if (propertyDescriptor == null) {
        logger.warn(s"External property defined for unknown property $propertyName on ${bci.getId}.")
      } else {
        try {
          val value = provider.resolve(externalProperty, propertyDescriptor.isPassword)
          logger.debug(s"Found $value for $propertyName in $provider.")
          propertyDescriptor.set(bci, value)
        } catch {
          case e: IllegalExternalPropertyReferenceException =>
            logger.warn(s"Provider $provider rejected to resolve for $propertyName.", e)
        }
      }
    }
  }

  private def asBaseConfigurationItem(ci: ConfigurationItem)(block: BaseConfigurationItem => Unit): Unit = {
    ci match {
      case bci: BaseConfigurationItem => block(bci)
      case _ => //Nothing
    }
  }

}
