package com.xebialabs.deployit.service.externalproperties.validation

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, ExternalProperty}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.lookup.{IllegalLookupValueProviderKeyException, LookupValueKey}
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.deployit.service.externalproperties.{ExternalValueProvider, IllegalExternalPropertyReferenceException}
import grizzled.slf4j.Logging
import javax.xml.bind.DatatypeConverter

import scala.jdk.CollectionConverters._
import scala.util.Try

class LookupValueKeyValidator(providers: Array[ExternalValueProvider]) extends ExternalPropertiesValidator with Logging {

  private val ListPattern = """(.*?[^,])*""".r
  private val MapEntryPattern = """(.*:.*)""".r

  override def canValidate(ep: ExternalProperty): Boolean = {
    ep.isInstanceOf[LookupValueKey]
  }

  override def validate(ci: ConfigurationItem): List[ValidationMessage] = {
    var validationMessages: List[ValidationMessage] = List()
    ci match {
      case bci: BaseConfigurationItem =>
        bci
          .get$externalProperties()
          .asScala
          .filter(ep => canValidate(ep._2))
          .foreach(ep => {
            val lookupValueKey = ep._2.asInstanceOf[LookupValueKey]
            val provider = providers.find(_.supports(lookupValueKey))
              .getOrElse(throw new IllegalArgumentException(s"No external value provider can handle LookupValueKey."))
            if (provider.isProviderAvailable(lookupValueKey)) {
              val propertyDescriptor = bci.getType.getDescriptor.getPropertyDescriptor(ep._1)
              if (propertyDescriptor == null) {
                logger.warn(s"External property defined for unknown property ${propertyDescriptor.getName} on ${bci.getId}.")
              } else {
                valueValidation(provider, lookupValueKey, propertyDescriptor)
                  .foreach(msg => validationMessages ::= ValidationMessage.warn(bci.getId, ep._1, msg))
              }
            }
            else {
              logger.debug(s"Could not use '${lookupValueKey.getProviderId}' for resolving value. Check if it is accessible or configured correctly.")
              validationMessages ::= ValidationMessage.warn(bci.getId, ep._1,
                s"Provider '${lookupValueKey.getProviderId}' cannot be accessed to resolve value for field '${ep._1}'")
            }
          })
      case _ =>
    }
    validationMessages
  }

  private def valueValidation(
                               provider: ExternalValueProvider, lookupValueKey: LookupValueKey,
                               propertyDescriptor: PropertyDescriptor): List[String] = {
    var warningMessages: List[String] = List()
    try {
      val value = provider.resolve(lookupValueKey, propertyDescriptor.isPassword)
      if (value != null) {
        logger.debug(s"Found value for field '${propertyDescriptor.getName}' in '${lookupValueKey.getProviderId}'.")
        value match {
          case str: String =>
            warningMessages :::= validateValueType(propertyDescriptor, str)
          case _ =>
            warningMessages ::= s"Unexpected lookup value type for field '${propertyDescriptor.getName}'."
        }
      }
      else {
        logger.debug(s"Did not find value for '${propertyDescriptor.getName}' in '${lookupValueKey.getProviderId}'" +
          s" - value fetch validated with false result.")
        warningMessages ::= s"Can't assign value to field '${propertyDescriptor.getName}'. Provider '${lookupValueKey.getProviderId}' " +
          s"does not contain value for key '${lookupValueKey.getKey}'."
      }
    }
    catch {
      case e: IllegalExternalPropertyReferenceException =>
        logger.warn(s"Provider '${lookupValueKey.getProviderId}' rejected to resolve value for field '${propertyDescriptor.getName}'.", e)
        warningMessages ::= s"Provider '${lookupValueKey.getProviderId}' rejected to resolve value for '${propertyDescriptor.getName}'"
      case e: IllegalLookupValueProviderKeyException =>
        logger.warn(s"Provider '${lookupValueKey.getProviderId}' rejected to resolve value for field '${propertyDescriptor.getName}'. Reason: ${e.getMessage}", e)
        warningMessages ::= s"Provider '${lookupValueKey.getProviderId}' rejected to resolve value for field '${propertyDescriptor.getName}'. \nReason: ${e.getMessage}"
    }
    warningMessages
  }

  private def validateValueType(propertyDescriptor: PropertyDescriptor, stringValue: String): List[String] = {
    var warningMessages: List[String] = List()
    propertyDescriptor.getKind match {
      case PropertyKind.BOOLEAN =>
        if(Try(stringValue.toBoolean).isFailure) {
          warningMessages ::= s"Can't assign fetched lookup value type to boolean field '${propertyDescriptor.getName}'."
        }
      case PropertyKind.INTEGER =>
        if(Try(stringValue.toInt).isFailure) {
          warningMessages ::= s"Can't assign fetched lookup value type to integer field '${propertyDescriptor.getName}'."
        }
      case PropertyKind.DATE =>
        if(Try(DatatypeConverter.parseDate(stringValue)).isFailure) {
          warningMessages ::= s"Can't assign fetched lookup value type to date field '${propertyDescriptor.getName}'."
        }
      case PropertyKind.MAP_STRING_STRING =>
        if (!stringValue.split(',').forall {
            case MapEntryPattern(_) => true
            case _ => false
          }) warningMessages ::= s"Can't assign fetched lookup value type to map of string field '${propertyDescriptor.getName}'."
      case PropertyKind.SET_OF_STRING | PropertyKind.LIST_OF_STRING =>
        stringValue match {
          case ListPattern(_) => ()
          case _ => warningMessages ::= s"Can't assign fetched lookup value type to collection of string field '${propertyDescriptor.getName}'."
        }
      case PropertyKind.ENUM =>
        if (!propertyDescriptor.getEnumValues.asScala.exists(_.equalsIgnoreCase(stringValue))) {
          warningMessages ::= s"Can't assign fetched lookup value type to enum of string field '${propertyDescriptor.getName}'."
        }
      case _ => ()
    }
    warningMessages
  }

}
