package com.xebialabs.plugin.zip

import java.util.zip.{ZipEntry, ZipInputStream}
import java.io.InputStream
import scala.util.matching.Regex

trait ZipSupport {

  def inputStreamForEntry(zip: ZipInputStream, entryName: String) = {
    findEntryByName(zip, entryName).map(entry => new ZipEntryInputStream(zip)).getOrElse {
      throw new RuntimeException(s"entry $entryName not found")
    }
  }

  def inputStreamForEachEntryMatching(regex: Regex, zipInputStream: ZipInputStream)(`do`: (InputStream, ZipEntry) => Unit) {
    var entry: ZipEntry = null
    do {
      entry = zipInputStream.getNextEntry
      if (entry != null && regex.findFirstIn(entry.getName).isDefined) {
        `do`(new ZipEntryInputStream(zipInputStream), entry)
      }
    } while (entry != null)
    None
  }

  def findEntry(zip: ZipInputStream)(filter: ZipEntry => Boolean): Option[ZipEntry] = {
    var entry: ZipEntry = zip.getNextEntry
    while (entry != null) {
      if (filter(entry)) return Option(entry)
      entry = zip.getNextEntry
    }
    None
  }

  def findEntryByName(zip: ZipInputStream, name: String) = findEntry(zip) { entry =>
    entry.getName == name || entry.getName == name + "/"
  }

  protected def containsEntryName(zipInputStream: ZipInputStream, entryName: String) = findEntryByName(zipInputStream, entryName).isDefined
}

private[zip] class ZipEntryInputStream(inputStream: ZipInputStream) extends InputStream {

  override def read(): Int = {
    if (inputStream.available() == 1) {
      inputStream.read()
    } else {
      -1
    }
  }

  override def close(): Unit = inputStream.close()
}