package com.xebialabs.plugin.classloader

import java.io.{File, FilenameFilter, InputStream}
import java.net.URL
import java.util

import com.xebialabs.plugin.protocol.xlp.PluginURL
import com.xebialabs.plugin.zip.PluginScanner
import com.xebialabs.xlplatform.utils.PerformanceLogging
import de.schlichtherle.truezip.file.TFile

object PluginClassLoader {
  def apply(pluginExtension: String, pluginDirectory: File, parentClassLoader: ClassLoader) =
    new PluginClassLoader(pluginExtension, Seq(pluginDirectory), Seq(), parentClassLoader)
}

private[plugin] class PluginClassLoader(pluginExtension: String, pluginDirectories: Iterable[File], explodedDirectories: Iterable[File], parentClassLoader: ClassLoader)
  extends ClassLoader(parentClassLoader) with PluginScanner with PerformanceLogging {

  import scala.collection.convert.wrapAsJava._

  override def findClass(name: String): Class[_] = logWithTime(s"Loading class $name") {
    logger.trace(s"Loading class $name")
    val classOption = findResourceUrl(convertClassName(name)).map(loadClassFromUrl(name, _))
    classOption.getOrElse(throw new ClassNotFoundException(s"$name not found"))
  }

  private def loadClassFromUrl(className: String, resourceUrl: URL): Class[_] = {
    logger.trace(s"Loading class from url $resourceUrl")
    val classInputStream = resourceUrl.openStream()
    try {
      val bytes: Array[Byte] = readFully(classInputStream)
      if (bytes.isEmpty) {
        throw new ClassFormatError("Could not load class. Empty stream returned")
      }
      definePackageIfNeeded(className)
      val clazz = defineClass(className, bytes, 0, bytes.size)
      resolveClass(clazz)
      clazz
    } finally {
      classInputStream.close()
    }
  }

  private def definePackageIfNeeded(className: String): Unit = {
    val packageName: String = className.split('.').init.mkString(".")
    Option(getPackage(packageName)).getOrElse(definePackage(packageName, null, null, null, null, null, null, null))
  }

  override def findResource(name: String): URL = logWithTime(s"Loading resource $name")(findResourceUrl(name).orNull)

  override def findResources(name: String): util.Enumeration[URL] = logWithTime(s"Loading resources $name")(resourcesByName(name).map(PluginURL(_)).toIterator)

  private def findResourceUrl(name: String): Option[URL] = resourceByName(name)

  private def convertClassName(className: String) = className.replaceAll("\\.", "/").concat(".class")

  private def listAllContent(root: TFile): Stream[TFile] = {
    if (root.isFile && !root.isArchive) {
      Seq(root).toStream
    } else {
      val fileStream = {
        for {
          r <- Option(root)
          children <- Option(r.listFiles())
        } yield children.toStream
      }.getOrElse(Stream.empty)
      fileStream ++ fileStream.flatMap(listAllContent)
    }
  }

  private def resourceByName(resourcePath: String): Option[URL] = {
    import collection.convert.wrapAll._
    Option(ResourceFinder.findResourceInDirs(resourcePath, allJarLibraries.toList))
  }

  private def resourcesByName(resourcePath: String): Stream[TFile] = {
      val r = for {
        library <- allJarLibraries
        allRes = listAllContent(library)
        resource <- allRes if filterByResourceName(resource, resourcePath)
      } yield resource

      r.toSet.toStream
  }

  private def filterByResourceName(resource: TFile, resourcePath: String): Boolean = {
    if (resource.getEnclEntryName == null) {
      resource.getName == resourcePath
    } else {
      resource.getEnclEntryName == resourcePath
    }
  }

  private def readFully(is: InputStream) = Stream.continually(is.read).takeWhile(-1 != _).map(_.toByte).toArray

  private val allJarLibraries: Iterable[TFile] = {
    val archives = pluginDirectories.flatMap { pluginDirectory =>
      val jarPlugins       = findAllPluginFiles(pluginDirectory, "jar").map(new TFile(_))
      val extensionPlugins = findAllPluginFiles(pluginDirectory, pluginExtension).map(new TFile(_)).flatMap(_.listFiles(jarFilter))
      jarPlugins ++ extensionPlugins
    }

    archives foreach (_.listFiles()) //To mount every jar file to avoid it mounting in recursion

    explodedDirectories.map(new TFile(_)) ++ archives
  }

  private lazy val jarFilter = new FilenameFilter {
    override def accept(dir: File, name: String): Boolean = name.endsWith(".jar")
  }
}
