package com.xebialabs.plugin.classloader

import java.io.{ByteArrayOutputStream, File, InputStream}
import java.net.URL
import java.util
import java.util.jar.{JarFile, JarInputStream}
import java.util.zip.{ZipEntry, ZipFile}
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.plugin.classloader.PluginClassLoader.{ExplodedPlugin, JarPlugin, Plugin, XLPlugin}
import com.xebialabs.plugin.protocol.xlp.{JarURL, XlpURL}
import com.xebialabs.plugin.zip.PluginScanner
import com.xebialabs.xlplatform.utils.{PerformanceLogging, ResourceManagement}
import grizzled.slf4j.Logging
import org.slf4j.LoggerFactory

import scala.annotation.tailrec
import scala.collection.mutable
import scala.jdk.CollectionConverters._

object PluginClassLoader {
  private[PluginClassLoader] val hotfixLogger = LoggerFactory.getLogger("hotfix")

  implicit class NormalizePath(val path: String) extends AnyVal {
    def toUnixPath: String = path.replace(File.separatorChar, '/')
  }

  def apply(pluginExtension: String, pluginDirectory: File, parentClassLoader: ClassLoader) =
    new PluginClassLoader(pluginExtension, Seq(pluginDirectory), Seq(), parentClassLoader)

  private[PluginClassLoader] trait Plugin {
    def getResources(name: String): Seq[URL]
  }

  private[PluginClassLoader] case class JarPlugin(file: File) extends Plugin {
    val jarFile = new JarFile(file)

    override def getResources(name: String): Seq[URL] = {
      Option(jarFile.getEntry(name)).map(e => Seq(JarURL(file.getAbsolutePath, e.getName))).getOrElse(Seq())
    }
  }

  private[this] case class XLPluginNode(name: String, subNodes: mutable.Map[String, XLPluginNode], url: URL)

  private[PluginClassLoader] case class XLPlugin(file: File) extends Plugin with Logging {
    val zipFile = new ZipFile(file)
    lazy val embeddedJars: List[ZipEntry] = {
      zipFile.entries().asScala.collect {
        case e if e.getName.endsWith(".jar") => e
      }.toList
    }

    def buildNode(filePath: String, libraryName: String, remainingParts: List[String], prefix: String, node: XLPluginNode): Unit = {
      remainingParts match {
        case Nil =>
        case head :: tail =>
          val newPrefix = s"$prefix/$head"
          val headNode = node.subNodes.get(head) match {
            case Some(value) => value
            case None =>
              val newNode = XLPluginNode(head, mutable.Map(), XlpURL(filePath, libraryName, newPrefix))
              node.subNodes(head) = newNode
              newNode
          }
          buildNode(filePath, libraryName, tail, newPrefix, headNode)
      }
    }

    lazy val pluginMap: Map[String, XLPluginNode] = {
      var plugins: Map[String, XLPluginNode] = Map()
      val filePath = file.getAbsolutePath.toUnixPath
      embeddedJars.foreach { ze =>
        val libraryName = ze.getName.toUnixPath.replace("./", "")
        val rootNode = XLPluginNode("/", mutable.Map(), XlpURL(filePath, libraryName, "/"))
        val stream = zipFile.getInputStream(ze)
        ResourceManagement.using(new JarInputStream(stream)) { jis =>
          var entry = jis.getNextJarEntry
          while (entry != null) {
            val n = entry.getName.toUnixPath
            val parts = n.split('/').toList
            buildNode(filePath, libraryName, parts, "/", rootNode)
            entry = jis.getNextJarEntry
          }
        }
        plugins += ((libraryName, rootNode))
      }
      plugins
    }

    override def getResources(name: String): Seq[URL] = {
      val parts = name.split('/').toList
      @tailrec
      def getUrlForPath(node: XLPluginNode, parts: List[String]): Option[URL] = {
        parts match {
          case Nil => Some(node.url)
          case head :: tail =>
            node.subNodes.get(head) match {
              case Some(value) => getUrlForPath(value, tail)
              case None => None
            }
        }
      }
      pluginMap.values.flatMap{ node =>
        getUrlForPath(node, parts)
      }
    }.toSeq
  }

  private[PluginClassLoader] case class ExplodedPlugin(explodedDir: File) extends Plugin {
    override def getResources(name: String): Seq[URL] = {
      val f = new File(explodedDir, name)
      if (f.exists()) {
        Seq(f.toURI.toURL)
      } else Seq()
    }
  }

}


class PluginClassLoader(pluginExtension: String, pluginDirectories: Iterable[File], explodedDirectories: Iterable[File], parentClassLoader: ClassLoader)
  extends ClassLoader(parentClassLoader) with PluginScanner with PerformanceLogging {

  override def findClass(name: String): Class[_] = logWithTime(s"Loading class $name") {
    val str: Option[URL] = findResourceUrl(convertClassName(name))
    val classOption = str.map(loadClassFromUrl(name, _))
    classOption.getOrElse(
      throw new ClassNotFoundException(
        s"""A plugin could not be loaded due to a missing class ($name). Please remove the offending plugin to successfully start the server.
           |Classes related to JCR were removed from the server because of the migration from JCR to SQL.
           |If the plugin depends on these classes and its functionality is required, please contact support to fix your configuration.
           |$name not found""".stripMargin.replaceAll("\n", ""))
    )
  }

  private def loadClassFromUrl(className: String, resourceUrl: URL): Class[_] = {
    logger.trace(s"Loading class from url $resourceUrl")
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    using(resourceUrl.openStream()) { classInputStream =>
      val bytes: Array[Byte] = readFully(classInputStream)
      if (bytes.isEmpty) {
        throw new ClassFormatError("Could not load class. Empty stream returned")
      }
      definePackageIfNeeded(className)
      val clazz = defineClass(className, bytes, 0, bytes.length)
      resolveClass(clazz)
      clazz
    }
  }

  private def definePackageIfNeeded(className: String): Unit = {
    val packageName: String = className.split('.').init.mkString(".")
    Option(getPackage(packageName)).getOrElse(definePackage(packageName, null, null, null, null, null, null, null))
  }

  def logHotfix(url: URL): URL = {
    if (url != null && url.toString.contains("hotfix")) {
      PluginClassLoader.hotfixLogger.warn(s"Loading class/resource from hotfix: $url")
    }
    url
  }

  override def findResource(name: String): URL = logWithTime(s"Loading resource $name")(logHotfix(findResourceUrl(name).orNull))

  override def findResources(name: String): util.Enumeration[URL] = logWithTime(s"Loading resources $name")({
    resourcesByName(name).map(u => {
      logger.trace(s"Found $u for $name")
      u
    }).iterator.asJavaEnumeration
  })

  private def findResourceUrl(name: String): Option[URL] = resourceByName(name)

  private def convertClassName(className: String) = className.replace('.', '/').concat(".class")

  private def resourceByName(resourcePath: String): Option[URL] = {
    resourcesByName(resourcePath).headOption
  }

  private def resourcesByName(resourcePath: String): Seq[URL] = {
    classPathRoots.flatMap(_.getResources(resourcePath)).toSeq
  }

  private def readFully(is: InputStream) = {
    val os = new ByteArrayOutputStream()
    OverthereUtils.write(is, os)
    os.toByteArray
  }

  private val classPathRoots: Iterable[Plugin] = {
    val archives = pluginDirectories.flatMap { pluginDirectory =>
      val jarPlugins = findAllPluginFiles(pluginDirectory, "jar").map(JarPlugin)
      val extensionPlugins = findAllPluginFiles(pluginDirectory, pluginExtension).map(XLPlugin)
      jarPlugins ++ extensionPlugins
    }
    explodedDirectories.map(ExplodedPlugin) ++ archives
  }
}
