package com.xebialabs.plugin.manager.repository.nexus

import java.io.InputStream
import java.nio.file.Path

import com.xebialabs.plugin.manager.metadata.{ArtifactId, Version}
import com.xebialabs.plugin.manager.{PluginId, requireProperFilename}


// For Uploading
case class NexusArtifact(content: Either[Path, Either[InputStream, String]], metadata: ArtifactMetadata, classifier: Option[String] = None)

object NexusArtifact {
  def apply(path: Path, metadata: ArtifactMetadata, classifier: Option[String]): NexusArtifact =
    new NexusArtifact(Left(path), metadata, classifier)

  def apply(content: String, metadata: ArtifactMetadata, classifier: Option[String]): NexusArtifact =
    new NexusArtifact(Right(Right(content)), metadata, classifier)

  def apply(input: InputStream, metadata: ArtifactMetadata, classifier: Option[String]): NexusArtifact =
    new NexusArtifact(Right(Left(input)), metadata, classifier)
}


// For Searching
case class ArtifactLink(extension: String, classifier: Option[String]) {
  requireProperFilename(extension, "extension")
  classifier.foreach(requireProperFilename(_, "classifier"))
}

case class ArtifactHit(repositoryId: String, artifactLinks: Seq[ArtifactLink]) {
  requireProperFilename(repositoryId, "repositoryId")
}

case class ArtifactMetadata(groupId: String, artifactId: String, version: Version, packaging: String) {
  requireProperFilename(groupId, "groupId")
  requireProperFilename(artifactId, "artifactId")
  requireProperFilename(packaging, "packaging")
}

object ArtifactMetadata {
  def apply(groupId: String, artifactId: String, version: String, packaging: String): ArtifactMetadata = {
    val parsed = Version.fromString(version)
    require(parsed.nonEmpty, s"Invalid version format: $version")
    new ArtifactMetadata(groupId, artifactId, parsed.get, packaging)
  }

  def fromArtifactId(id: ArtifactId, packaging: String): ArtifactMetadata =
    ArtifactMetadata(id.groupId, id.artifactId, id.version, packaging)

  implicit class ArtifactMetadataOps(val metadata: ArtifactMetadata) extends AnyVal {
    def fileName: String = s"${metadata.artifactId}-${metadata.version.id}.${metadata.packaging}"
    def id: String = s"${metadata.groupId}:${this.fileName}"

    def desc: String = s"${metadata.groupId}:${metadata.artifactId}:${metadata.version.id}:${metadata.packaging}"

    def toPluginId(repositoryId: String): PluginId.Artifact = {
      PluginId.Artifact(repositoryId, metadata.groupId, metadata.artifactId, metadata.version, metadata.packaging, None)
    }
  }
}

case class ArtifactResult(groupId: String,
                          artifactId: String,
                          version: Version,
                          latestRelease: String,
                          latestReleaseRepositoryId: String,
                          artifactHits: Seq[ArtifactHit]) {
  requireProperFilename(groupId, "groupId")
  requireProperFilename(artifactId, "artifactId")
}

object ArtifactResult {
  val supportedExtensions = Seq("jar")

  implicit class ArtifactResultOps(val result: ArtifactResult) extends AnyVal {

    def allExtensions: Map[String, ArtifactMetadata] = {
      for {
        hit <- result.artifactHits
        link <- hit.artifactLinks
      } yield {
        link.extension -> ArtifactMetadata(result.groupId, result.artifactId, result.version, link.extension)
      }
    }.toMap
    def extensions: Map[String, ArtifactMetadata] = allExtensions.filterKeys(supportedExtensions.contains)

    def withExtension(ext: String): Option[ArtifactMetadata] = extensions.get(ext)

    def metadata: Option[ArtifactMetadata] = extensions.headOption.map(_._2)

    def toPluginId(repositoryId: String): Option[PluginId.Artifact] = extensions.get("jar").map(_.toPluginId(repositoryId))

    def pluginId(repositoryId: String): PluginId.Artifact =
      PluginId.Artifact.parsed(repositoryId, result.groupId, result.artifactId, result.version)

    def id: String = s"${result.groupId}:${result.artifactId}-${result.version}"
  }

}

case class RepoDetails(repositoryId: String,
                       repositoryName: String,
                       repositoryContentClass: String,
                       repositoryKind: String,
                       repositoryPolicy: String,
                       repositoryURL: String)

case class NexusArtifactResults(totalCount: Int,
                                from: Int,
                                count: Int,
                                tooManyResults: Boolean,
                                collapsed: Boolean,
                                repoDetails: Seq[RepoDetails],
                                data: Seq[ArtifactResult])