package com.xebialabs.plugin.manager.repository.sql

import com.xebialabs.deployit.server.api.upgrade.Version
import org.springframework.jdbc.core.JdbcTemplate

import java.sql.{Connection, ResultSet}
import scala.jdk.CollectionConverters.ListHasAsScala
import scala.util.Using

/**
  * Repository for XL_VERSION table. The table and columns case (UPPER vs lower) is different across DBs and products so
  * DB metadata is inspected to create correct queries.
  *
  * See VersionRepositoryTest for examples.
  *
  * @param jdbcTemplate
  */
class VersionRepository(jdbcTemplate: JdbcTemplate) {

  lazy val dialect: Dialect = Dialect(jdbcTemplate)

  private lazy val selectFromXlVersion = createSelectQuery

  def selectComponentVersion(componentName: String): Option[Version] =
  jdbcTemplate.query(selectFromXlVersion, (rs: ResultSet, row: Int) => {
    Version.valueOf(rs.getString(1), rs.getString(2))
  }, componentName)
    .asScala.headOption

  def xlVersionTableExists: Boolean = {
    Using(jdbcTemplate.getDataSource.getConnection()) { connection =>
      doesTableExist(connection, "XL_VERSION") || doesTableExist(connection, "xl_version")
    }.get
  }

  def createSelectQuery: String = {
    val isTableUpperCase = Using(jdbcTemplate.getDataSource.getConnection()) { connection =>
      doesTableExist(connection, XL_VERSION.TABLE)
    }.get

    val isColumnUpperCase = Using(jdbcTemplate.getDataSource.getConnection()) { connection =>
      val tableName = if (isTableUpperCase) XL_VERSION.TABLE else XL_VERSION.TABLE.toLowerCase
      doesColumnExist(connection, tableName, XL_VERSION.component)
    }.get

    val xlVersion = createName(XL_VERSION.TABLE, isTableUpperCase)
    val component = createName(XL_VERSION.component, isColumnUpperCase)
    val version = createName(XL_VERSION.version, isColumnUpperCase)

    s"SELECT $component, $version FROM $xlVersion WHERE $component = ?"
  }

  private def createName(name: String, toUpperCase: Boolean) = {
    val nameCased = if (toUpperCase) name else name.toLowerCase
    dialect.quoteIfNecessary(nameCased)
  }

  private def doesTableExist(connection: Connection, tableName: String) = {
    connection.getMetaData.getTables(null, null, tableName, null).next()
  }

  private def doesColumnExist(connection: Connection, tableName: String, columnName: String) = {
    connection.getMetaData.getColumns(null, null, tableName, columnName).next()
  }

  abstract class Table(val TABLE: String)

  object XL_VERSION extends Table("XL_VERSION") {
    val component: String = "COMPONENT"
    val version: String = "VERSION"
  }

}
