package com.xebialabs.plugin.manager.repository.config

import com.xebialabs.plugin.manager.config.Repository
import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.repository.config.ListMethod.ByGroupId
import com.xebialabs.plugin.manager.{PluginId, requireProperFilename}
import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json.{DefaultJsonProtocol, JsObject, JsonWriter, enrichAny}

import scala.util.Try

abstract class PluginRepositoryConfig(val server: PluginServerConfig,
                                      val productVersion: Version,
                                      val repositoryId: String,
                                      val metadataArtifactId: MetadataId,
                                      val listMethod: ByGroupId) {

  lazy val metadataVersion: Version = productVersion.copy(extra = None)
}

sealed trait ListMethod

object ListMethod {

  case class ByGroupId(groupId: String, classifier: Option[String]) extends ListMethod {
    requireProperFilename(groupId, "groupId")
  }

  object ByGroupId {
    def fromConfig(repositoryConfig: Repository): Try[ByGroupId] = {
      Try(Option(repositoryConfig.getNexusGroupId).get).map { groupId =>
        ByGroupId(groupId, None)
      }
    }
  }

}

case class MetadataId(groupId: String, artifactId: String) {
  requireProperFilename(groupId, "groupId")
  requireProperFilename(artifactId, "artifactId")
}

object MetadataId {

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
    implicit val metadataIdWriter: JsonWriter[MetadataId] = metadataId =>
      JsObject(
        "group-id" -> metadataId.groupId.toJson,
        "artifact-id" -> metadataId.artifactId.toJson
      )
  }

  implicit class MetadataIdOps(val metadataId: MetadataId) extends AnyRef {
    def id: String = s"${metadataId.groupId}:${metadataId.artifactId}"

    def toArtifact(repoName: String, version: Version, packaging: String, classifier: Option[String] = None): PluginId.Artifact =
      PluginId.Artifact.parsed(repoName, metadataId.groupId, metadataId.artifactId, version, packaging, classifier)
  }

}

