package com.xebialabs.plugin.manager

import com.xebialabs.plugin.manager.LocalPluginManager.listDir
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.model.DbPlugin
import com.xebialabs.plugin.manager.rest.api.PluginSource.LOCAL
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import grizzled.slf4j.Logging

import java.nio.file.{Files, Path, Paths}
import scala.concurrent.ExecutionContext
import scala.jdk.CollectionConverters._


object LocalPluginManager {
  val name = "__local__"

  def listDir(dir: Path): LazyList[Path] = {
    if (dir.toFile.isDirectory) {
      using(Files.newDirectoryStream(dir)) { list =>
        list.asScala.to(LazyList).force
      }
    } else {
      LazyList.empty
    }
  }

}

object LocalPluginManagerFactory {
  val defaultPluginsPath: Path = Paths.get("plugins")
  var pluginsPath: Path = defaultPluginsPath

  def resetPluginsPath(): Unit = pluginsPath = defaultPluginsPath

  def createInstance(repositoryGroupId: Map[String, String], sqlPluginRepository: SqlPluginRepository): LocalPluginManager = {
    new LocalPluginManager(repositoryGroupId, sqlPluginRepository, pluginsPath)
  }
}

class LocalPluginManager(repositoryGroupId: Map[String, String],
                         sqlPluginRepository: SqlPluginRepository,
                         pluginsDir: Path)
  extends PluginManager with Logging {

  def this(repositoryGroupId: Map[String, String], sqlPluginRepository: SqlPluginRepository) = {
    this(repositoryGroupId, sqlPluginRepository, LocalPluginManagerFactory.pluginsPath)
  }

  def name: String = LocalPluginManager.name

  implicit val ec: ExecutionContext = ExecutionContext.global

  def ensurePluginsDirectoryExists(): Unit = {
    pluginsDir.resolve(name).toFile.mkdirs()
  }

  override def listInstalled(): Seq[PluginId] = {
    ensurePluginsDirectoryExists()

    val pluginsInDb = sqlPluginRepository.getAllWithoutBytes
    val pluginsOnFs = listDir(pluginsDir).flatMap(listInstalledIn)

    val pluginsInDbNames = pluginsInDb.map(_.name)
    val filteredFsPlugins = pluginsOnFs.filterNot(fsPlugin => pluginsInDbNames.contains(fsPlugin.name))

    pluginsInDb.map(toPluginId).concat(filteredFsPlugins)
  }

  override def listInstalledOnFilesystem(): Seq[PluginId] = {
    ensurePluginsDirectoryExists()
    listDir(pluginsDir).flatMap(listInstalledIn)
  }

  override def installOrUpdate(plugin: Plugin): Unit = {
    logger.debug(s"install(${plugin.id.id}, ${plugin.metadata})")

    try {
      val dbPlugins = sqlPluginRepository.getByName(plugin.id.name)

      val dbPlugin = DbPlugin(plugin)
      if (dbPlugins.size == 1) {
        sqlPluginRepository.update(dbPlugin, dbPlugins.head)
      } else {
        sqlPluginRepository.insert(dbPlugin)
      }

    } catch {
      case e: Exception => logger.error(s"Error installing plugin ${plugin.id.id} to the database with error ${e.getMessage}")
      // TODO handle better
    }
  }

  override def install(plugin: Plugin): Unit = {
    logger.debug(s"install(${plugin.id.id}, ${plugin.metadata})")
    try {
      val dbPlugin = DbPlugin(plugin)
      sqlPluginRepository.insert(dbPlugin)
    } catch {
      case e: Exception =>
        logger.error(s"Error installing plugin ${plugin.id.id} to the database with error ${e.getMessage}")
        throw e
    }
  }

  override def update(existingPlugin: PluginId, newVersion: Plugin): Unit = {
    logger.debug(s"update(${existingPlugin.id}, $existingPlugin), with ${newVersion.id.id}, ${newVersion.metadata}")
    val newDbPlugin = DbPlugin(newVersion)
    val dbPlugins = sqlPluginRepository.getByName(existingPlugin.name)
    sqlPluginRepository.update(newDbPlugin, dbPlugins.head)
  }

  override def getPluginBy(repositoryId: PluginSource.Value, groupId: String, artifactId: String, version: Option[String]): Option[DbPlugin] = {
    sqlPluginRepository.getPluginBy(repositoryId, groupId, artifactId, version)
  }

  override def fetchPluginStatuses(): Map[String, PluginStatus.Value] = {
    sqlPluginRepository.getAllWithoutBytes.map(plugin => plugin.name -> plugin.installationStatus).toMap
  }

  override def uninstall(plugin: DbPlugin): Boolean = {
    plugin.installationStatus.equals(PluginStatus.READY_FOR_INSTALL) && sqlPluginRepository.delete(plugin) > 0
  }

  override def revertPreviousVersionIfNecessary(plugin: DbPlugin): Unit = {
    val pluginsOnFs = listDir(pluginsDir).flatMap(listInstalledIn)
    val fsPluginPreviousVersion = pluginsOnFs.filter(_.name.equalsIgnoreCase(plugin.name))
    if (fsPluginPreviousVersion.isEmpty) {
      debug(s"Nothing to revert from the filesystem..")
    } else if (fsPluginPreviousVersion.size > 1) {
      error(s"Found two or more plugins with identical name ${plugin.name} on the filesystem, aborting revert operation..")
    } else {
      sqlPluginRepository.insert(toDbPluginWithInstalledStatus(fsPluginPreviousVersion.head))
    }
  }

  protected def listInstalledIn(dir: Path): Seq[PluginId] =
    if (dir.getFileName.toString == name) {
      listLocal(dir)
    } else {
      listRepository(dir)
    }

  protected def listLocal(localDir: Path): LazyList[PluginId] =
    listPluginDirectory(localDir) { filePath =>
      Option(PluginId.LocalFile(filePath))
    }

  protected def listRepository(repositoryDir: Path): LazyList[PluginId] = {
    val repository = repositoryDir.getFileName.toString
    repositoryGroupId.get(repository).map { groupId =>
      listPluginDirectory(repositoryDir) { filePath =>
        val filename = filePath.getFileName.toString
        PluginId.Artifact.fromIdString(s"$repository:$groupId:$filename")
      }
    }.getOrElse(LazyList.empty)
  }

  protected def listPluginDirectory(path: Path)(f: Path => Option[PluginId]): LazyList[PluginId] =
    listDir(path)
      .filter(p => ConfigWrapper.isValidExtension(p.getFileName.toString))
      .flatMap(f)

  private def toPluginId(dbPlugin: DbPlugin): PluginId = dbPlugin match {
    case DbPlugin(_, name, version, extension, _, _, LOCAL, _, _, _) => LocalFile(name, version.flatMap(v => Version.fromString(v)), extension)
    case DbPlugin(_, name, version, extension, groupId, _, source, _, _, _) => Artifact(source.toString, groupId, name, version.orNull, extension)
  }

  private def toDbPluginWithInstalledStatus(plugin: PluginId): DbPlugin = {
    def bytes = Files.readAllBytes(pluginsDir.resolve(plugin.path).resolve(plugin.filename))

    val dbPlugin = DbPlugin(Plugin(plugin, None, bytes))
    dbPlugin.copy(installationStatus = PluginStatus.INSTALLED)
  }

}
