package com.xebialabs.plugin.manager.repository.artifactory

import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.exception.PluginRepositoryException._
import com.xebialabs.plugin.manager.metadata.{ArtifactId, ExtendedMetadata, MetadataEntry, Version}
import com.xebialabs.plugin.manager.repository.artifactory.ArtifactoryArtifactResult.extractArtifactInfo
import com.xebialabs.plugin.manager.repository.config.ListMethod
import com.xebialabs.plugin.manager.repository.config.ListMethod.ByGroupId
import com.xebialabs.plugin.manager.repository.protocol.ArtifactResult
import com.xebialabs.plugin.manager.repository.storage.{PluginMetadataMemoryStorage, PluginMetadataStorage}
import com.xebialabs.plugin.manager.repository.{EntityOps, HttpPluginRepository, RespSyntax, latestVersion, verifyPlugin}
import grizzled.slf4j.Logging
import org.apache.pekko.http.scaladsl.model.HttpCharsets.`UTF-8`
import org.apache.pekko.http.scaladsl.model.HttpMethods.GET
import org.apache.pekko.http.scaladsl.model.MediaTypes.{`application/json`, `application/octet-stream`, `application/zip`}
import org.apache.pekko.http.scaladsl.model.headers.{Accept, Authorization, RawHeader, `Accept-Charset`}
import org.apache.pekko.http.scaladsl.model.{HttpRequest, Uri}

import java.io.ByteArrayInputStream
import java.nio.charset.StandardCharsets.UTF_8
import scala.concurrent.Future
import scala.util.{Failure, Success}
import spray.json._


case class ArtifactoryPluginRepository(name: String,
                                       config: ArtifactoryRepositoryConfig,
                                       cache: PluginMetadataStorage)
  extends HttpPluginRepository(name, config, cache)
    with ArtifactoryJsonProtocol
    with MetadataEntry.Protocol
    with Logging {


  override def searchFromRemote(query: Option[String]): Future[Seq[ArtifactResult]] = {
    logger.info(s"$name: Searching for '${query.getOrElse("")}'")
    for {
      resp <- searchArtifact(query)
      content <- resp
        .onSuccess(_.asJson[ArtifactoryArtifactResults])
        .onFailure((status, msg) => QueryFailed(query, status, msg))
      plugins <- {
        Future.successful(
          content.results.filter(_.ext.nonEmpty)
        )
      }
    } yield {
      logger.info(s"$name: Found ${plugins.size} plugins (query: '${query.getOrElse("")}')")
      plugins.flatMap(plugin => extractArtifactInfo(plugin))
    }
  }


  protected def getRequest(pluginId: PluginId.Artifact): HttpRequest = {
    val groupPath = pluginId.groupId.split('.').foldLeft[Uri.Path](Uri.Path.Empty)(_ / _)
    val downloadUri = config.server.uri
      .withPath(config.server.serviceUri.path / config.repositoryId ++
        groupPath / pluginId.artifactId / pluginId.artifactVersion.id / pluginId.filename
      )
    HttpRequest(GET,
      uri = downloadUri,
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/zip`, `application/octet-stream`, `application/json`)
      ))
  }


  protected def searchRequest(query: Uri.Query): HttpRequest = {
    HttpRequest(GET,
      uri = config.server.uri
        .withPath(config.server.serviceUri.path ++ ArtifactoryPluginRepository.searchPath)
        .withQuery(query.+:("specific", "true")),
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/json`),
        RawHeader("X-Result-Detail", "info")
      )
    )
  }


  def searchLatestMetadataArtifact(packaging: String, classifier: Option[String]): Future[PluginId.Artifact] = {
    logger.debug(s"$name: Searching latest version of ${config.metadataArtifactId.id}, classifier: $classifier, packaging: $packaging")
    for {
      getAllVersions <- searchMetadata(packaging, classifier)
      allVersions <- getAllVersions
        .onSuccess(_.asJson[ArtifactoryArtifactResults])
        .onFailure((status, msg) => SearchMetadataFailed(config.metadataArtifactId, packaging, classifier, status, msg))
        .map(result => {
          result.results.collect {
            case r: ArtifactoryArtifactResult if r.ext contains packaging =>
              Version.fromString(r.version)
          }.flatten
        })

      artifact <- Future.fromTry {
        latestVersion(config.metadataVersion)(allVersions) match {
          case None =>
            logger.debug(s"$name: Could not find any candidate for metadata artifact ${config.metadataArtifactId.id}${classifier.fold("")("-" + _)}.$packaging")
            Failure(NoMetadataVersions(config.metadataArtifactId, packaging, classifier, config.metadataVersion, allVersions))
          case Some(version) =>
            Success(config.metadataArtifactId.toArtifact(name, version, packaging, classifier))
        }
      }
    } yield {
      logger.debug(s"$name: Found metadata artifact: ${artifact.id()}")
      artifact
    }
  }

  override protected def searchParams(method: ListMethod): Map[String, Option[String]] = Map("repos" -> Some(config.repositoryId)) ++ {
    method match {
      case g: ByGroupId => Map(
        "g" -> Some(g.groupId),
        "c" -> g.classifier
      )
    }
  }

  override def getFromRemote(pluginId: PluginId.Artifact): Future[ByteArrayInputStream] = {
    logger.info(s"$name: Downloading plugin ${pluginId.id()}")
    for {
      getPlugin <- downloadArtifact(pluginId)
      pluginContent <- getPlugin
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(pluginId, status, msg))

      getSha1 <- downloadSha1(pluginId)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(pluginId, status, msg))

      (sha1, length) <- verifyPlugin(pluginId, sha1Content, pluginContent, getPlugin.entity.contentLengthOption)
    } yield {
      logger.info(s"$name: Downloaded plugin from ${config.server.uri}: ${pluginId.id()} ($length bytes) | SHA1: $sha1")
      new ByteArrayInputStream(pluginContent)
    }
  }

  override def fetchMetadata: Future[Map[ArtifactId, ExtendedMetadata]] = {
    logger.info(s"Downloading metadata for plugin repository $name from ${config.server.uri}")
    for {
      jsonId <- searchLatestMetadataArtifact("json", None)

      resp <- downloadArtifact(jsonId)
      content <- resp
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(jsonId, status, msg))

      getSha1 <- downloadSha1(jsonId)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(jsonId, status, msg))

      (sha1, length) <- verifyPlugin(jsonId, sha1Content, content, resp.entity.contentLengthOption)
    } yield {
      val entries = new String(content, UTF_8)
        .parseJson
        .convertTo[Seq[MetadataEntry]]
      logger.debug(s"$name: Downloaded metadata from ${config.server.uri}: ${jsonId.id()} ($length bytes) | SHA1: $sha1 | ${entries.size} entries")
      entries.map(e => e.artifactId.copy(repository = Some(name)) -> e.metadata).toMap
    }
  }

  override def fetchLogos: Future[ByteArrayInputStream] = {
    logger.debug(s"$name: Downloading logos")
    for {
      logoArtifact <- searchLatestMetadataArtifact("zip", Some("data"))

      resp <- downloadArtifact(logoArtifact)
      content <- resp
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(logoArtifact, status, msg))

      getSha1 <- downloadSha1(logoArtifact)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(logoArtifact, status, msg))

      (sha1, length) <- verifyPlugin(logoArtifact, sha1Content, content, resp.entity.contentLengthOption)
    } yield {
      logger.debug(s"$name: Downloaded metadata logos data from ${config.server.uri}: ${logoArtifact.id()} ($length bytes) | SHA1: $sha1")
      new ByteArrayInputStream(content)
    }
  }

}

object ArtifactoryPluginRepository {
  val searchPath: Uri.Path = Uri.Path.Empty / "api" / "search" / "gavc"

  def memCached(name: String, config: ArtifactoryRepositoryConfig): ArtifactoryPluginRepository = {
    new ArtifactoryPluginRepository(name, config, new PluginMetadataMemoryStorage(name))
  }

}
