package com.xebialabs.plugin.manager

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginVersionProperties, Version, VersionExpr}
import com.xebialabs.plugin.manager.rest.api.PluginSource
import com.xebialabs.plugin.manager.util.PluginFileUtils
import grizzled.slf4j.Logging
import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._

import java.io.File
import java.nio.file.{Path, Paths}
import scala.util.Try
import scala.util.matching.Regex

sealed trait PluginId {
  def name: String

  def version: Option[Version]

  def extension: String

  def source: PluginSource.Value

  def groupId: String

  // Need to override in Artifact because classifier is part of it
  def filename(): String = {
    val optionalVersion = version.map(v => s"-${v.id}").getOrElse("")
    s"""$name$optionalVersion.$extension"""
  }

  def pluginFilePath(pluginsDir: Path): String = {
    List(pluginsDir, source, filename()).mkString(File.separator)
  }

  def id(): String = {
    s"$source:$groupId:${filename()}"
  }

  override def toString: String = {
    filename()
  }
}

object PluginId extends Logging {
  val localGroupId: String = LocalPluginManager.name
  lazy val extensionsPattern: String = ConfigWrapper.getAnyExtensionsRegexPattern()
  lazy val pattern: Regex = raw"""(.+)-(\d.*)\.($extensionsPattern)""".r
  lazy val patternNoVersion: Regex = raw"""(.+)\.($extensionsPattern)""".r
  lazy val patternVersionWithNightly: Regex = raw"""(.+)-(\d.*-\d.*)\.($extensionsPattern)""".r
  lazy val patternWithWildcard: Regex = raw"""(.+)\*\.($extensionsPattern)""".r

  case class LocalFile private(name: String, version: Option[Version], extension: String) extends PluginId {
    override def source: PluginSource.Value = PluginSource.LOCAL

    override def groupId: String = localGroupId
  }

  case class Artifact private(repository: String,
                              groupId: String,
                              artifactId: String,
                              artifactVersion: Version,
                              packaging: String,
                              classifier: Option[String]) extends PluginId {

    requireProperFilename(repository, "repository")
    require(repository != LocalPluginManager.name, s"repository ${LocalPluginManager.name} is reserved for manually installed plugins")
    requireProperFilename(groupId, "groupId")
    require(groupId != localGroupId, s"$localGroupId is reserved for manually installed plugins")
    requireProperFilename(artifactId, "artifactId")
    private val classifier1: Option[String] = classifier.filter(_.nonEmpty)
    require(classifier1.fold(true)(c => {
      require(!c.matches("\\s+"), "Classifier cannot contain spaces")
      true
    }))

    override def name: String = artifactId

    override def version: Option[Version] = Option(artifactVersion)

    override def extension: String = packaging

    override def source: PluginSource.Value = PluginSource.withName(repository)

    override def filename(): String = {
      s"$artifactId-${artifactVersion.id}${classifier.map("-" + _).getOrElse("")}.$packaging"
    }
  }

  object LocalFile {
    private lazy val idPattern: Regex = s"""${LocalPluginManager.name}:$localGroupId:${pattern.toString}""".r
    private lazy val idPatternNoVersion: Regex = s"""${LocalPluginManager.name}:$localGroupId:${patternNoVersion.toString}""".r
    private lazy val idPatternVersionWithNightly: Regex = s"""${LocalPluginManager.name}:$localGroupId:${patternVersionWithNightly.toString}""".r
    private lazy val idPatternWithWildcard: Regex = s"""${LocalPluginManager.name}:$localGroupId:${patternWithWildcard.toString}""".r

    private def getLocalPluginFromFilename(filename: String, extension: String): LocalFile = {
      filename match {
        case patternVersionWithNightly(basename, versionWithNightly, _) =>
          new LocalFile(basename, Option(versionWithNightly).flatMap(Version.fromString), extension)
        case pattern(basename, versionString, _) =>
          new LocalFile(basename, Option(versionString).flatMap(Version.fromString), extension)
        case patternNoVersion(basename, _) =>
          new LocalFile(basename, None, extension)
        case _ =>
          new LocalFile(filename.stripSuffix(extension), None, extension)
      }
    }

    private def getLocalFile(filename: String)(zipPluginCallback: () => PluginVersionProperties): LocalFile = {
      requireProperFilename(filename, "Filename")
      val extension = PluginFileUtils.getPluginExtension(filename)

      if (extension == ConfigWrapper.EXTENSION_ZIP) {
        // extract plugin name and version from plugin-version.properties for new zip based plugin
        val pluginVersionProperties = zipPluginCallback()
        new LocalFile(pluginVersionProperties.plugin, Option(pluginVersionProperties.version).flatMap(Version.fromString), extension)
      } else {
        getLocalPluginFromFilename(filename, extension)
      }
    }

    def apply(filename: String, bytes: Array[Byte]): LocalFile = {
      getLocalFile(filename)(() => PluginFileUtils.getPluginPropertiesForZip(filename, bytes))
    }

    def apply(filePath: Path): LocalFile = {
      val filename = filePath.getFileName.toString
      getLocalFile(filename)(() => PluginFileUtils.getPluginPropertiesForZip(filePath))
    }

    def apply(basename: String, version: String, extension: String): LocalFile =
      parsed(basename, Version.fromString(version), extension)

    def parsed(basename: String, version: Option[Version], extension: String): LocalFile = {
      requireProperFilename(basename, "basename")
      if (!ConfigWrapper.getExtensions().contains(extension)) {
        throw new IllegalArgumentException(s"Extension must be '${ConfigWrapper.getExtensions().map(ext => s".${ext}").mkString(" or ")}'")
      }
      new LocalFile(basename, version, extension)
    }

    def fromIdString(id: String): Option[LocalFile] = id match {
      case idPatternVersionWithNightly(base, ver, extension) =>
        for {
          basename <- Option(base)
          version = Option(ver).getOrElse("")
          parsed <- Try(LocalFile(basename, version, extension)).toOption
        } yield parsed
      case idPattern(base, ver, extension) =>
        for {
          basename <- Option(base)
          version = Option(ver).getOrElse("")
          parsed <- Try(LocalFile(basename, version, extension)).toOption
        } yield parsed
      case idPatternWithWildcard(base, extension) =>
        for {
          basename <- Option(base)
          parsed <- Try(LocalFile.apply(basename, None, extension)).toOption
        } yield parsed
      case idPatternNoVersion(base, extension) =>
        for {
          basename <- Option(base)
          parsed <- Try(LocalFile.apply(basename, None, extension)).toOption
        } yield parsed
      case _ =>
        None
    }

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
      with Version.Protocol {
      val pluginIdLocalFileWriter: RootJsonWriter[LocalFile] = {
        case id@LocalFile(base, version, extension) => JsObject(
          "id" -> id.id().toJson,
          "filename" -> id.filename().toJson,
          "repository" -> LocalPluginManager.name.toJson,
          "basename" -> base.toJson,
          "version" -> (version: Option[Version]).toJson,
          "extension" -> extension.toJson
        )
      }
      private val pluginIdLocalFileReader: RootJsonReader[LocalFile] = { jsValue =>
        val maybeLocalFile = jsValue match {
          case obj: JsObject =>
            obj.getFields("id") match {
              case Seq(JsString(id)) => PluginId.LocalFile.fromIdString(id)
              case _ => None
            }
          case JsString(id) => PluginId.LocalFile.fromIdString(id)
          case _ => None
        }
        maybeLocalFile.getOrElse {
          deserializationError(s"Cannot parse PluginId.LocalFile from '$jsValue'")
        }
      }
      implicit val pluginIdLocalFormat: RootJsonFormat[LocalFile] = rootJsonFormat(pluginIdLocalFileReader, pluginIdLocalFileWriter)
    }

  }

  object Artifact {
    private lazy val idPattern: Regex = s"""(.+):(.+):$pattern""".r
    private lazy val idPatternWithNightly: Regex = s"""(.+):(.+):$patternVersionWithNightly""".r

    def apply(repository: String, groupId: String, artifactId: String, version: String, extension: String, classifier: Option[String] = None): Artifact = {
      requireProperFilename(version, "version")
      val parsedVersion = Version.fromString(version)
      require(parsedVersion.nonEmpty, s"Invalid version format: $version")
      parsed(repository, groupId, artifactId, parsedVersion.get, extension, classifier)
    }

    def parsed(repository: String, groupId: String, artifactId: String, version: Version, extension: String, classifier: Option[String] = None): Artifact = {
      new Artifact(repository, groupId, artifactId, version, extension, classifier.filter(_.nonEmpty))
    }

    def fromIdString(id: String): Option[Artifact] = id match {
      case idPatternWithNightly(repo, groupId, artifactId, version, extension) =>
        Some(Artifact.apply(repo, groupId, artifactId, version, extension, None))
      case idPattern(repo, groupId, artifactId, version, extension) =>
        Some(Artifact.apply(repo, groupId, artifactId, version, extension, None))
      case _ => None
    }

    trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
      with Version.Protocol {
      private val pluginIdArtifactReader: RootJsonReader[Artifact] = { jsValue =>
        val maybeArtifactId = jsValue match {
          case obj: JsObject =>
            obj.getFields("id") match {
              case Seq(JsString(id)) => PluginId.Artifact.fromIdString(id)
              case _ => None
            }
          case JsString(id) => PluginId.Artifact.fromIdString(id)
          case _ => None
        }
        maybeArtifactId.getOrElse {
          deserializationError(s"Cannot parse PluginId.Artifact from '$jsValue'")
        }
      }
      val pluginIdArtifactWriter: RootJsonWriter[Artifact] = {
        case id@Artifact(repo, groupId, artifactId, version, packaging, _) => JsObject(
          "id" -> id.id().toJson,
          "filename" -> id.filename().toJson,
          "repository" -> repo.toJson,
          "groupId" -> groupId.toJson,
          "artifactId" -> artifactId.toJson,
          "packaging" -> packaging.toJson,
          "version" -> version.toJson
        )
      }
      implicit val pluginIdArtifactFormat: RootJsonFormat[Artifact] = rootJsonFormat(pluginIdArtifactReader, pluginIdArtifactWriter)
    }

    implicit class ArtifactOps(val artifact: Artifact) extends AnyVal {
      def toArtifactId: ArtifactId = ArtifactId(artifact.groupId, artifact.artifactId, artifact.artifactVersion, artifact.packaging, Some(artifact.repository))
    }

  }

  def fromIdString(id: String): Option[PluginId] = {
    if (id.startsWith(LocalPluginManager.name + ":")) {
      LocalFile.fromIdString(id)
    } else {
      Artifact.fromIdString(id)
    }
  }

  implicit class PluginIdOps(val pluginId: PluginId) extends AnyVal {

    def fold[B](ifLocal: LocalFile => B, ifArtifact: Artifact => B): B = pluginId match {
      case local: LocalFile => ifLocal(local)
      case artifact: Artifact => ifArtifact(artifact)
    }

    def toArtifactId: ArtifactId = fold(
      l => ArtifactId(PluginId.localGroupId, l.name, l.version.getOrElse(Version.zero), l.extension),
      a => new Artifact.ArtifactOps(a).toArtifactId
    )

    def path: Path = Paths.get(pluginId.source.toString)
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with VersionExpr.Protocol
    with LocalFile.Protocol
    with Artifact.Protocol {
    private val pluginIdWriter: RootJsonWriter[PluginId] = {
      case l: LocalFile => pluginIdLocalFileWriter.write(l)
      case a: Artifact => pluginIdArtifactWriter.write(a)
    }
    private val pluginIdReader: RootJsonReader[PluginId] = { jsValue =>
      val maybePluginId = jsValue match {
        case obj: JsObject =>
          obj.getFields("id") match {
            case Seq(JsString(id)) => PluginId.fromIdString(id)
            case _ => None
          }
        case JsString(id) => PluginId.fromIdString(id)
        case _ => None
      }
      maybePluginId.getOrElse {
        deserializationError(s"Cannot parse PluginId from '$jsValue'")
      }
    }
    implicit val pluginIdFormat: RootJsonFormat[PluginId] = rootJsonFormat(pluginIdReader, pluginIdWriter)
  }
}


