package com.xebialabs.plugin.manager.repository.artifactory

import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.repository.protocol.{ArtifactMetadata, ArtifactResult}

import java.io.InputStream
import java.nio.file.Path


// For Uploading
case class ArtifactoryArtifact(content: Either[Path, Either[InputStream, String]], metadata: ArtifactMetadata, classifier: Option[String] = None)

object ArtifactoryArtifact {
  def apply(path: Path, metadata: ArtifactMetadata, classifier: Option[String]): ArtifactoryArtifact =
    new ArtifactoryArtifact(Left(path), metadata, classifier)

  def apply(content: String, metadata: ArtifactMetadata, classifier: Option[String]): ArtifactoryArtifact =
    new ArtifactoryArtifact(Right(Right(content)), metadata, classifier)

  def apply(input: InputStream, metadata: ArtifactMetadata, classifier: Option[String]): ArtifactoryArtifact =
    new ArtifactoryArtifact(Right(Left(input)), metadata, classifier)
}



case class ArtifactoryArtifactResults(results: Seq[ArtifactoryArtifactResult])

case class ArtifactoryArtifactResult(downloadUri: String,
                                     ext: String,
                                     version: String,
                                     classifier: Option[String])


object ArtifactoryArtifactResult {

  def extractArtifactInfo(artifact: ArtifactoryArtifactResult): Option[ArtifactResult] = {
    val artifactPathPattern = ".*(/com/xebialabs/xlrelease/.*)".r
    val artifactPath = artifact.downloadUri match {
      case artifactPathPattern(path) => path
      case _ =>
        return None
    }

    // Split the path by "/" and remove the leading empty string from the split
    val parts = artifactPath.split("/").tail

    // Ensure that there are at least 3 parts to extract groupId and artifactId
    if (parts.length < 3) {
      None
    } else {
      val groupId = parts.dropRight(3).mkString(".")
      val artifactId = parts.lift(parts.length - 3)

      // Extract version using Option to handle potential failure
      val versionOption = Version.fromString(artifact.version)

      (artifactId, versionOption) match {
        case (Some(aId), Some(version)) =>
          Some(ArtifactResult(
            groupId = groupId,
            artifactId = aId,
            version = version,
            latestRelease = "",
            latestReleaseRepositoryId = "",
            artifactHits = Seq.empty
          ))
        case _ => None
      }
    }
  }

}
