package com.xebialabs.plugin.manager.repository.nexus

import com.xebialabs.plugin.manager.config.{NexusMetadataArtifact, Repository}
import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.repository.config.{ListMethod, MetadataId, PluginRepositoryConfig, PluginServerConfig}
import com.xebialabs.plugin.manager.service.ProductConfig
import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._

import scala.util.{Failure, Success, Try}

case class NexusRepositoryConfig(override val server: NexusServerConfig,
                                 override val productVersion: Version,
                                 override val repositoryId: String,
                                 override val metadataArtifactId: MetadataId,
                                 override val listMethod: ListMethod.ByGroupId)
  extends PluginRepositoryConfig(server, productVersion, repositoryId, metadataArtifactId, listMethod)


object NexusRepositoryConfig {

  def byGroupId(server: NexusServerConfig,
                productVersion: Version,
                repositoryId: String,
                metadataId: MetadataId,
                groupId: String,
                classifier: Option[String] = None): NexusRepositoryConfig = {
    NexusRepositoryConfig(server, productVersion, repositoryId, metadataId, ListMethod.ByGroupId(groupId, classifier))
  }


  def fromConfig(name: String)(repositoryConfig: Repository, servers: Map[String, Try[PluginServerConfig]])
                (implicit productConfig: ProductConfig): Try[NexusRepositoryConfig] = {

    def metadataIdFromConfig(config: NexusMetadataArtifact): Try[MetadataId] = for {
      groupId <- Try(Option(config.getGroupId).get)
      artifactId <- Try(Option(config.getArtifactId).get)
    } yield MetadataId(groupId, artifactId)

    for {
      _ <- Try(Option(repositoryConfig.getEnabled).get).flatMap {
        case "true" => Success(())
        case _ => Failure(new IllegalArgumentException(s"Repository configuration '$name' is not enabled."))
      }

      server <- if (repositoryConfig.getServerRef != null && repositoryConfig.getServerRef.nonEmpty) {
        val serverRefName = repositoryConfig.getServerRef
        servers.get(serverRefName) match {
          case None => Failure(new RuntimeException(s"servers.$serverRefName is missing."))
          case Some(serverConfig) => serverConfig
        }
      } else {
        Failure(new IllegalArgumentException(s"Repository configuration '$name' is missing a server-ref property."))
      }
      listMethod <- ListMethod.ByGroupId.fromConfig(repositoryConfig)
      repositoryId <- Try(Option(repositoryConfig.getNexusRepositoryId).get)
      metadataId <- Try(Option(repositoryConfig.getNexusMetadataArtifact).get).flatMap(metadataIdFromConfig)
    } yield {
      NexusRepositoryConfig(server.asInstanceOf[NexusServerConfig], productConfig.version, repositoryId, metadataId, listMethod)
    }
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol
    with NexusServerConfig.Protocol
    with MetadataId.Protocol {

    implicit val nexusConfigWriter: RootJsonWriter[NexusRepositoryConfig] = config =>
      JsObject(
        "server" -> config.server.toJson,
        "repository-id" -> config.repositoryId.toJson,
        "nexus-metadata-artifact" -> config.metadataArtifactId.toJson,
        "repository-type" -> "nexus-by-group-id".toJson,
        "nexus-group-id" -> config.listMethod.groupId.toJson
      )
  }
}

