package com.xebialabs.plugin.manager.util

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.exception.{InvalidPluginVersionPropertiesException, MissingPluginVersionPropertiesException}
import com.xebialabs.plugin.manager.model.FilePlugin
import com.xebialabs.plugin.manager.rest.api.PluginSource
import com.xebialabs.plugin.manager.rest.api.PluginSource.{LOCAL, XLD_OFFICIAL, XLR_OFFICIAL}
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import grizzled.slf4j.Logging

import java.io.File
import java.nio.file.{Files, Path}
import scala.jdk.CollectionConverters.IteratorHasAsScala
import scala.util.{Failure, Success, Using}

object PluginsTraverser extends Logging {

  def listPlugins(dir: File): List[FilePlugin] = {
    dir.listFiles()
      .filter(file => ConfigWrapper.isValidExtension(file.toString))
      .map(filePath => FilePlugin(filePath.toPath))
      .toList
  }

  def traverse(dir: Path)(officialCallback: (PluginSource.Value, List[FilePlugin]) => Unit, localCallback: List[FilePlugin] => Unit): Unit = {
    Using(Files.list(dir)) { path =>
      path.filter(_.isDirectory)
        .map(dir => PluginSet(PluginSource.withName(dir.getFileName.toString), listPlugins(dir)))
        .forEach(pluginSet => {
          pluginSet.source match {
            case XLD_OFFICIAL | XLR_OFFICIAL =>
              officialCallback(pluginSet.source, pluginSet.plugins)
            case LOCAL =>
              localCallback(pluginSet.plugins)
            case _ => error(s"Could not match plugins directory(source) value ${pluginSet.source.toString}, skipping..")
          }
        })
    } match {
      case Success(_) =>
        info(s"Successfully synced plugins")
      case Failure(err) =>
        error(s"Error during plugin sync with message: ${err.getMessage}")
        err match {
          case _: InvalidPluginVersionPropertiesException | _: MissingPluginVersionPropertiesException => throw err
          case _ => ()
        }
    }
  }

  def listOfficialFsPlugins(dir: Path): List[FilePlugin] = {
    Using(Files.list(dir)) { paths => paths.iterator().asScala.toList
        .filter(_.isDirectory)
        .map(dir => PluginSet(PluginSource.withName(dir.getFileName.toString), listPlugins(dir)))
        .find(pluginSet => List(XLD_OFFICIAL, XLR_OFFICIAL).contains(pluginSet.source))
        .map(_.plugins).get
    }.get
  }
}

case class PluginSet(source: PluginSource.Value, plugins: List[FilePlugin])