package com.xebialabs.plugin.manager.compatibility;


import org.jboss.windup.decompiler.api.DecompilationException;
import org.jboss.windup.decompiler.api.DecompilationFailure;
import org.jboss.windup.decompiler.api.DecompilationListener;
import org.jboss.windup.decompiler.api.DecompilationResult;
import org.jboss.windup.decompiler.fernflower.FernFlowerResultSaver;
import org.jboss.windup.decompiler.fernflower.FernflowerDecompiler;
import org.jboss.windup.decompiler.fernflower.FernflowerJDKLogger;
import org.jboss.windup.decompiler.util.Filter;
import org.jboss.windup.util.exception.WindupStopException;
import org.jetbrains.java.decompiler.main.Fernflower;
import org.jetbrains.java.decompiler.main.extern.IBytecodeProvider;
import org.jetbrains.java.decompiler.main.extern.IFernflowerPreferences;
import org.jetbrains.java.decompiler.util.InterpreterUtil;
import java.util.logging.Logger;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

/**
 * Copied decompileArchiveImpl method code from https://github.com/windup/windup/blob/6.3.3.Final/decompiler/impl-fernflower/src/main/java/org/jboss/windup/decompiler/fernflower/FernflowerDecompiler.java,
 * and removed lines 231 and 232 to enable de-compilation of internal classes.
 */
public class XlFernflowerDecompiler extends FernflowerDecompiler {

    private static final Logger LOG = Logger.getLogger(XlFernflowerDecompiler.class.getName());

    @Override
    public DecompilationResult decompileArchiveImpl(Path archive, Path outputDir, Filter<ZipEntry> filter, final DecompilationListener delegate)
            throws DecompilationException {

        final DecompilationResult result = new DecompilationResult();
        DecompilationListener listener = new DecompilationListener() {
            private boolean cancelled;

            @Override
            public void fileDecompiled(List<String> inputPaths, String outputPath) {
                try {
                    result.addDecompiled(inputPaths, outputPath);
                    delegate.fileDecompiled(inputPaths, outputPath);
                } catch (WindupStopException stop) {
                    this.cancelled = true;
                    throw new WindupStopException(stop);
                }
            }

            @Override
            public void decompilationFailed(List<String> inputPath, String message) {
                result.addFailure(new DecompilationFailure(message, inputPath, null));
                delegate.decompilationFailed(inputPath, message);
            }

            @Override
            public void decompilationProcessComplete() {
                delegate.decompilationProcessComplete();
            }

            @Override
            public boolean isCancelled() {
                return this.cancelled;
            }
        };

        LOG.info("Decompiling archive '" + archive.toAbsolutePath() + "' to '" + outputDir.toAbsolutePath() + "'");
        final JarFile jar;
        try {
            jar = new JarFile(archive.toFile());
        } catch (IOException ex) {
            throw new DecompilationException("Can't load .jar: " + archive, ex);
        }

        try {
            final AtomicInteger jarEntryCount = new AtomicInteger(0);
            Enumeration<JarEntry> countEnum = jar.entries();
            while (countEnum.hasMoreElements()) {
                countEnum.nextElement();
                jarEntryCount.incrementAndGet();
            }

            final Enumeration<JarEntry> entries = jar.entries();

            while (entries.hasMoreElements()) {
                final JarEntry entry = entries.nextElement();

                final String name = entry.getName();

                if (!name.endsWith(".class")) {
                    jarEntryCount.decrementAndGet();
                    continue;
                }

                if (filter != null) {
                    Filter.Result filterRes = filter.decide(entry);

                    if (filterRes == Filter.Result.REJECT) {
                        jarEntryCount.decrementAndGet();
                        continue;
                    } else if (filterRes == Filter.Result.STOP) {
                        break;
                    }
                }

                IBytecodeProvider bytecodeProvider = new IBytecodeProvider() {
                    @Override
                    public byte[] getBytecode(String externalPath, String internalPath) throws IOException {
                        return InterpreterUtil.getBytes(jar, entry);
                    }
                };

                FernFlowerResultSaver resultSaver = getResultSaver(Collections.singletonList(entry.getName()), outputDir.toFile(), listener);
                Fernflower fernflower = new Fernflower(bytecodeProvider, resultSaver, getOptions(),new FernflowerJDKLogger());
                fernflower.getStructContext().addSpace(new File(entry.getName()), true);
                fernflower.decompileContext();

                if (!resultSaver.isFileSaved())
                    listener.decompilationFailed(Collections.singletonList(entry.getName()), "File was not decompiled!");
            }
            listener.decompilationProcessComplete();
            return result;
        } finally {
            try {
                jar.close();
            } catch (IOException e) {
                LOG.warning("Failed to close jar file: " + jar.getName());
            }
        }
    }

    private FernFlowerResultSaver getResultSaver(final List<String> requests, File directory, final DecompilationListener listener) {
        return new FernFlowerResultSaver(requests, directory, listener);
    }

    private Map<String, Object> getOptions() {
        Map<String, Object> options = new HashMap<>();
        options.put(IFernflowerPreferences.MAX_PROCESSING_METHOD, 30);
        options.put(IFernflowerPreferences.DECOMPILE_GENERIC_SIGNATURES, "1");
        options.put(IFernflowerPreferences.BYTECODE_SOURCE_MAPPING, "1");
        return options;
    }

}
